<?php


/**
 * Classe responsavel por processar os retornos dos boletos bancarios seguindo o padrao CNAB400 ou CNAB240
 * e j? executa a baixa das parcelas.
 * O padrao CNAB400 utilizado eh do Banco Bradesco. (Modificado em 03/12/2010)
 * Alguns metodos foram baseados no projeto retorno-boletophp de Manoel Campos da Silva Filho.
 *
 * @author Pablo de Oliveira Lima - 06/2010
 */
class RetornoBoleto  extends Zend_Db_Table_Abstract
{
    protected $_name = TB_RETORNO_BOLETO;

    /**
     * @property string $banco  Caminho para o arquivo de retorno bancario
     */
    protected $banco;

    /**
     * @property string $arquivo  Caminho para o arquivo de retorno bancario
     */
    protected $arquivo;

    /**
     * @property int TAMANHO_LINHA Define a quantidade de caracteres da linha
     */
    protected $tamanho_linha;

    /**
     * @property Zend_Db_Table_Abstract $tb_parcela  Instancia da tabela de parcelas do banco de dados
     */
    protected $tb_parcela;

    /**
     * @property array $linhas  Contem a lista de pagamentos encontradas no retorno bancario
     */
    protected $pagamentos;

    /**
     * @property array $linhasErradas  Contem a lista de linhas encontradas erradas durante o processamento
     */
    protected $linhasErradas;

    /**
     * @property array $parcelasCorretas  Contem a lista de parcelas que foram processadas corretamente
     */
    protected $parcelasCorretas;

    /**
     * @property array $parcelasErradas  Contem a lista de parcelas que nao foram processadas corretamente
     */
    protected $parcelasErradas;

    /**
     * @property array $parcelasErradas  Contem a lista de parcelas que foram processadas mas possuem inconsistencias
     */
    protected $parcelasVerificar;

    /**
     * @property array $codigosMovimento  Contem a lista de status presentes nos registros
     */
    protected $codigosMovimento;

    /**
     * @property array $header_arquivo  Contem os dados do cabecalho do arquivo de retorno
     */
    protected $header_arquivo;

    /**
     * @property array $header_lote  Contem os dados do cabecalho do lote do arquivo de retorno
     */
    protected $header_lote;

    /**
     * @property array $trailer_arquivo  Contem os dados do trailer do arquivo de retorno
     */
    protected $trailer_arquivo;

    /**
     * @property array $trailer_lote  Contem os dados do trailer do lote do arquivo de retorno
     */
    protected $trailer_lote;
    /**
     * @static int HEADER_ARQUIVO Define o valor que identifica uma coluna do tipo HEADER DE ARQUIVO
     */
    const HEADER_ARQUIVO = 0;
    /**
     * @static int HEADER_LOTE Define o valor que identifica uma coluna do tipo HEADER DE LOTE
     */
    const HEADER_LOTE = 1;
    /**
     * @static int DETALHE Define o valor que identifica uma coluna do tipo DETALHE
     */
    const DETALHE_240 = 3;
    /**
     * @static int DETALHE Define o valor que identifica uma coluna do tipo DETALHE
     */
    const DETALHE_400 = 1;
    /**
     * @static int TRAILER_LOTE Define o valor que identifica uma coluna do tipo TRAILER DE LOTE
     */
    const TRAILER_LOTE = 5;
    /**
     * @static int TRAILER_ARQUIVO Define o valor que identifica uma coluna do tipo TRAILER DE ARQUIVO
     */
    const TRAILER_ARQUIVO = 9;

    public function __construct($arquivo, $padrao = 240, $comRegistro = false)
    {
        if (file_exists($arquivo)) {
            $this->arquivo = $arquivo;
        } else {
            throw new Exception('No foi possvel encontrar o arquivo.');
        }
        $this->tb_parcela = new Parcela();
        $this->tb_lote = new Lote();
        $this->tb_contrato = new Contrato();
        $this->pagamentos = array();
        $this->tamanho_linha = $padrao;
        $this->tb_cliente = new Db_Cliente();

    }

    /**
     * Obtem os dados do arquivo de retorno bancario, colocando as parcelas obtidas
     * do arquivo, como pagas na base de dados.
     * Adicionado parametro migradas no lugar de verificar fl_migracao da tb_cliente,
     * para verificar arquivo retorno com digito verificador do banco Itau.(13/03/2018)
     * tpRetorno, serve para verificar o tipo de retorno, se  normal ou migradas. N = Normal, M = Migradas
     * @param void
     * @return array $resultado  Contem os registros alterados, como tambem os
     * erros encontrados no arquivo.
     * @author Pablo de Oliveira Lima.
     */
    public function processar($comRegistro = false, $migradas = false, $bc = null, $header = false, $tpRetorno = 'N', $idBanco = null)
    {
        // Processa o arquivo para gerar a lista de pagamentos
        if ($this->arquivo)
            $this->processarArquivo();

        if ($idBanco) {
            $dadosBanco = (new Banco)->getBanco($idBanco);
            $this->header_arquivo['banco'] = $dadosBanco['cd_banco'];
            $this->header_arquivo['agencia'] = $dadosBanco['agencia'];
            $this->header_arquivo['dv_agencia'] = $dadosBanco['agencia_dv'];
            $this->header_arquivo['conta_corrente'] = ($dadosBanco['cd_banco'] == '104' || $dadosBanco['cd_banco'] == '741') ? $dadosBanco['cd_cedente'] : $dadosBanco['conta_corrente'];
            $this->header_arquivo['dv_conta'] = $dadosBanco['conta_corrente_dv'];
            $_POST['cd_banco'] = $dadosBanco['cd_banco'];

        } else {
            //Seta o banco de acordo com o header do arquivo
            $_POST['cd_banco'] = $this->header_arquivo['cod_banco'];

            if (!$_POST['cd_banco'])
                $_POST['cd_banco'] = $this->header_arquivo['banco'];
        }

        $banco = Banco::findBancoByInfo($this->header_arquivo);

        $resultado = array(
            'banco' => ($banco) ? true : false,
            'conta_corrente_com_dv' => $banco['conta_corrente'] . '-' . $banco['conta_corrente_dv'],
            'header' => $this->header_arquivo
        );

        if (!$header) {
            // Lista de pagamentos
            if (is_array($this->pagamentos)) {

                foreach ($this->pagamentos as $k => $pagamento) {
                    $dados = false;
                    if ($this->tamanho_linha == 240) {
                        //Remove o registro que contabiliza a taxa bancaria
                        if (str_replace(' ', '', $pagamento['nosso_numero']) == '') {
                            continue;
                        }
                        if ($_POST['cd_banco'] == Banco::BANCO_DO_BRASIL) {
                            $numero_boleto = substr($pagamento['nosso_numero'], 7, 10);
                        } elseif ($_POST['cd_banco'] == Banco::BANCO_CAIXA_ECONOMICA) {
                            $numero_boleto = substr($pagamento['nosso_numero'], 7, 12);
                        } elseif ($_POST['cd_banco'] == Banco::BANCO_SANTANDER) {
                            $numero_boleto = (substr($pagamento['numero_documento'], 5, 10) > 0) ? $pagamento['numero_documento'] : $pagamento['nosso_numero'];
                        } elseif ($_POST['cd_banco'] == Banco::BANCO_BRADESCO) {
                            $numero_boleto = substr($pagamento['nosso_numero'], 8, 11);
                        } elseif ($_POST['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO) {
                            $numero_boleto = substr($pagamento['nosso_numero'], 8, 11);
                        } elseif ($_POST['cd_banco'] == Banco::BANCO_UNIPRIME) {
                            $numero_boleto = substr($pagamento['nosso_numero'], 8, 11);
                        } elseif ($_POST['cd_banco'] == Banco::BANCO_ITAU) {
                            //Condicao introduzida para verificar nosso numero com digito verificador.
                            if ($migradas) {
                                $numero_boleto = substr($pagamento['nosso_numero'], 3, 9);
                            } else {
                                $numero_boleto = substr($pagamento['nosso_numero'], 3, 8);
                            }
                        } elseif ($_POST['cd_banco'] == Banco::BANCO_SICOOB || $_POST['cd_banco'] == Banco::BANCO_NORDESTE) {
                            $numero_boleto = trim($pagamento['nosso_numero']);
                        } elseif ($_POST['cd_banco'] == Banco::BANCO_SICREDI) {
                            $numero_boleto = substr($pagamento['nosso_numero'], 2, 6);
                        } elseif ($_POST['cd_banco'] == Banco::BANCO_BANPARA) {
                            $numero_boleto = substr($pagamento['nosso_numero'], 7, 12);
                        } elseif ($_POST['cd_banco'] == Banco::BANCO_SAFRA) {
                            $numero_boleto = trim($pagamento['nosso_numero']);
                        } elseif ($_POST['cd_banco'] == Banco::BANCO_CRESOL) {
                            $numero_boleto = trim($pagamento['nosso_numero']);
                        } elseif ($_POST['cd_banco'] == "085") {
                            $numero_boleto = $pagamento['nosso_numero'];
                        }elseif ($_POST['cd_banco'] == "136") {
                            $numero_boleto = trim($pagamento['nosso_numero']);
                        }

                        $valor_boleto = (int)substr($pagamento['valor_pago'], 0, 13) . '.' . substr($pagamento['valor_pago'], 13);

                        // Somando valor de tarifa com valor total pago apenas para ITAU
                        if ($_POST['cd_banco'] == Banco::BANCO_ITAU) {
                            $valor_tarifa = (int)substr($pagamento['valor_tarifa'], 0, 13) . '.' . substr($pagamento['valor_tarifa'], 13);
                            // Recriando a variavel valor_pago com a tarifa somada para ser utilizado no programa todo
                            $pagamento['valor_pago'] = str_pad(str_replace('.', '', $valor_boleto + $valor_tarifa), 15, '0', STR_PAD_LEFT);
                            $valor_boleto = $valor_boleto + $valor_tarifa;
                        }

                        $juros = (int)substr($pagamento['acrescimos'], 0, 13) . '.' . substr($pagamento['acrescimos'], 13);


                        $data_pagamento = substr($pagamento['data_ocorrencia'], 4, 4) . '-' .
                            substr($pagamento['data_ocorrencia'], 2, 2) . '-' .
                            substr($pagamento['data_ocorrencia'], 0, 2);

                        if ($pagamento['data_credito'] == '00000000') {
                            $pagamento['data_credito'] = $pagamento['data_ocorrencia'];
                        } else {
                            $data_credito = substr($pagamento['data_credito'], 4, 4) . '-' .
                                substr($pagamento['data_credito'], 2, 2) . '-' .
                                substr($pagamento['data_credito'], 0, 2);
                        }

                    } elseif ($this->tamanho_linha == 400) {
                        $numero_boleto = substr($pagamento['titulo_banco'], 1, 10);
                        $valor_boleto = $pagamento['valor_pago'];
                        $juros_mora = $pagamento['juros_mora'];
                        $data_pagamento = Helper::getInputDate($pagamento['data_ocorrencia']);
                        $data_credito = Helper::getInputDate($pagamento['data_credito']);
                    } else {
                        continue;
                    }

                    $mensagem = Db_TabelaAgrupada::findByTabelaId(15, $pagamento['cod_movimento']);
                    $this->codigosMovimento[$pagamento['cod_movimento']] = array(
                        'codigo' => $pagamento['cod_movimento'],
                        'mensagem' => $mensagem[0]['descricao']
                    );

                    if (!in_array($_POST['cd_banco'], ['756', '004', '748', '422', '341','001','136'])) {
                        $numero_boleto = substr($numero_boleto, 2);
                    }

                    //Verifica se e para pesquisar por id da parcela ou id de migracao
                    $idParcelaContrato = ($migradas) ? 'id_parcela_contrato_antigo' : 'id_parcela_contrato';

                    $idsBancos = $banco['id'];

                    //Verifica se o banco possui outras contas associadas, caso sim, pesquisa parcela associadas as uma dessas contas
                    //$bancos = (new Banco())->fetchAll('id_banco_conta = ' . $banco['id_banco_conta']);
                    $bancos = $this->tb_parcela->getAdapter()->select()
                        ->from(array('b' => TB_BANCO), 'b.*')
                        ->join(array('bc' => TB_BANCO_CONTA), 'b.id_banco_conta = bc.id_banco_conta', array('fl_boleto'))
                        ->where('b.id_banco_conta = ' . $banco['id_banco_conta'])
                        ->query()
                        ->fetchAll();
                    if ($bancos) {
                        foreach ($bancos as $b) {
                            $idsBancosAssociados[] = $b['id'];
                            if($b['fl_boleto']!='0'&&!$migradas){
                                $idParcelaContrato="nr_boleto";
                            }
                        }
                        
                        $idsBancos = implode(', ', $idsBancosAssociados);
                        $idsBancosAssociados = []; //Apagar o array para no duplicar
                    }
                    if ($_POST['cd_banco'] == "085") {
                        $numero_boleto = str_replace(' ', '', $numero_boleto);
                        $numero_boleto = substr($numero_boleto, 6, 19);
                        $numero_boleto = "0{$numero_boleto}";
                    }

                    //Busca parcelas
                    if (is_numeric($numero_boleto)) {

                        $arraydados = $this->tb_parcela->getAdapter()->select()
                            ->from(array('p' => TB_PARCELA), array('*','id_banco_boleto' => 'id_banco'))
                            ->join(array('c' => TB_CONTRATO), 'c.id = p.id_contrato', array('idContrato' => 'id', 'id_pessoa'))
                            ->join(array('l' => TB_LOTES), 'c.id_lote = l.id', array('l.id_empreendimento'))
                            ->join(array('pe' => TB_PESSOA), 'c.id_pessoa = pe.id', array('nomecliente' => 'pe.nm_pessoa'))
                            ->join(array('b' => TB_BANCO), 'b.id_banco_conta  = p.id_banco_boleto and b.id_empreendimento = l.id_empreendimento', array('id_banco' =>'b.id'))
                            ->where('b.id IN (' . $idsBancos . ')');
                        //verifica o tipo de retorno N = Retorno Boleto, M = Retorno Migradas
                        if ($tpRetorno == 'N') {
                            if ($banco['cd_banco'] == '748' || $banco['cd_banco'] == '741') {
                                $arraydados->where('p.nr_boleto = ?', $numero_boleto);
                            } else {
                                $arraydados->where('p.'.$idParcelaContrato.' = ?', $numero_boleto);
                            }

                        } else {

                            if ($banco['cd_banco'] == '748' || $banco['cd_banco'] == '741') {
                                $arraydados->where('p.id_parcela_contrato_antigo = "' . $numero_boleto . '"');
                            } else {
                                $arraydados->where('p.' . $idParcelaContrato . ' = ' . $numero_boleto);
                            }

                        }
                        $arraydados = $arraydados->group('p.id_parcela_contrato')->query()->fetchAll();

                        if (count($arraydados) == 0) {

                            $arraydados = $this->tb_parcela->getAdapter()->select()
                                ->from(array('p' => TB_PARCELA), '*')
                                ->join(array('c' => TB_CONTRATO), 'c.id = p.id_contrato', array('idContrato' => 'id', 'id_pessoa','id_banco'))
                                ->join(array('l' => TB_LOTES), 'c.id_lote = l.id', array('l.id_empreendimento'))
                                ->join(array('pe' => TB_PESSOA), 'c.id_pessoa = pe.id', array('nomecliente' => 'pe.nm_pessoa'))
                                ->where('c.id_banco IN (' . $idsBancos . ')');
                            //verifica o tipo de retorno N = Retorno Boleto, M = Retorno Migradas
                            if ($tpRetorno == 'N') {
                                if ($banco['cd_banco'] == '748' || $banco['cd_banco'] == '741') {
                                    $arraydados->where('p.nr_boleto = ?', $numero_boleto);
                                } else {
                                    $arraydados->where('p.'.$idParcelaContrato.' = ?', $numero_boleto);
                                }

                            } else {

                                if ($banco['cd_banco'] == '748' || $banco['cd_banco'] == '741') {
                                    $arraydados->where('p.id_parcela_contrato_antigo = "' . $numero_boleto . '"');
                                } else {
                                    $arraydados->where('p.' . $idParcelaContrato . ' = ' . $numero_boleto);
                                }

                            }
                            $arraydados = $arraydados->group('p.id_parcela_contrato')->query()->fetchAll();
                        }
                        //Caso o array seja maior do que 1, faz o filtro pelo nome do cliente
                        //Implementao para corrigir o erro no Multi Banco dadr baixa na parcela do cliente errado
                        if (count($arraydados) > 1) {
                            foreach ($arraydados as $item) {
                                //Caso o Cliente informado no arquivo .RET for o mesmo do contrato, dá baixa na parcela. Caso não, pula a parcela.
                                if(substr_compare(trim($item['nomecliente']),trim($pagamento['sacado']),0,40) == 0){
                                    $dados[0] = $item;
                                }
                            }
                            if(!$dados){
                                foreach ($arraydados as $item) {
                                    if((floatval($item['valor_titulo'])/100==$pagamento['dt_parcela'])&&($item['vencimento']==implode("",array_reverse(explode("-",$pagamento['dt_parcela']))))){
                                        $dados[0] = $item;
                                    }
                                }
                            }

                        } else {
                            $dados = $arraydados;
                        }
                    } else {
                        $dados = false;
                    }

                    if ($dados) {

                        $idContrato = $dados[0]['id_contrato'];
                        $cdBaixa = $dados[0]['id_parcela_contrato'];
                        $valorParcela = $dados[0]['vl_parcela'];

                        // Adicionando informacoes do contrato
                        $dados[0]['contrato'] = Contrato::findContrato($idContrato);

                        //Buscando o contrato -> lote -> quadra
                        $contrato = $this->tb_contrato->fetchRow("id = '" . $dados[0]['contrato']->id . "' ");
                        $lote = $this->tb_lote->fetchRow("id = '" . $contrato['id_lote'] . "' ");
                        $pagamento['quadra'] = $lote['quadra'];
                        $dados[0]['quadra'] = $lote['quadra'];

                        $parcela = $this->tb_parcela->fetchRow($this->tb_parcela->select()
                            ->where('id_parcela_contrato = ?', $cdBaixa));

                        //Verifica o codigo de movimento do registro para identificar se ? de baixa ou de instrucao
                        if (!($pagamento['cod_movimento'] == '06' || $pagamento['cod_movimento'] == '17')) {

                            if ($banco['cd_banco'] == '748' || $banco['cd_banco'] == '741') {
                                $parcela['id_parcela_contrato'] = $parcela['nr_boleto'];
                                $pagamento['nosso_numero'] = $parcela['nr_boleto'];

                            } elseif ($banco['cd_banco'] == '104') {
                                /**
                                 * Condio criada para remover o ltimo caracter do nosso_numero, caso o banco
                                 * seja CAIXA. Essa modificao efeta apenas a visualizao do arquivo pdf.
                                 */
                                $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 0, -1);
                            }

                            $dados[0]['erro'] = 'Resposta  instruo';

                            if ($_POST['exibirTodos'] == '0' && $pagamento['cod_movimento'] == '02') {
                                continue;

                            }
                            $this->parcelasVerificar[] = array_merge($pagamento, $dados[0]);

                            continue;
                        }

                        //Testa se valores estao todos de acordo, se nao estiverem de baixa mas coloca na lista de analise.
                        $dadosContrato = Contrato::findContratoById($idContrato);
                        if ($valorParcela > $valor_boleto) {
                            $flDesconto = '';
                            $dadosContrato['tp_parcela'] = $parcela['tp_parcela'];

                            switch ($dadosContrato['tp_parcela']) {
                                case 'N':
                                    $flDesconto = $dadosContrato['fl_desc_normal'];
                                    break;
                                case 'S':
                                    $flDesconto = $dadosContrato['fl_desc_sinal'];
                                    break;
                                case 'I':
                                    $flDesconto = $dadosContrato['fl_desc_intercalada'];
                                    break;
                                case 'C':
                                    $flDesconto = $dadosContrato['fl_desc_chave'];
                                    break;
                            }

                            if ($flDesconto == 'S') {
                                $dados[0]['erro'] = 'Tit. Pago - Valor Inferior';
                                $pagamento['verificar'] = true;
                            } else {
                                $vlDesconto = ($valorParcela - $valor_boleto);
                                $parcela->desconto = $vlDesconto;
                            }
                        }

                        if (Helper::date_diff($data_pagamento, $parcela['dt_parcela']) >= 0) {
                            if ($valorParcela < $valor_boleto) {
                                $dados[0]['erro'] = 'Tit. Pago - Valor Superior';
                                $pagamento['verificar'] = true;
                            }
                        }
                        if ($dados[0]['pago'] == 1) {

                            if ($banco['cd_banco'] == '748' || $banco['cd_banco'] == '741') {
                                $parcela['id_parcela_contrato'] = $parcela['nr_boleto'];
                                $pagamento['nosso_numero'] = $parcela['nr_boleto'];
                            }
                            $dados[0]['erro'] = 'Tit. Baixado - Verificar';
                            $this->parcelasErradas[] = array_merge($pagamento, $dados[0]);
                            continue;
                        }

                        // Calculando juros para o padrao CNAB400 Bradesco
                        if ($this->tamanho_linha == 400 && $comRegistro && $pagamento['cod_movimento'] != '06' && $pagamento['cod_movimento'] != '17') {

                            if ($banco['cd_banco'] == '748' || $banco['cd_banco'] == '741') {
                                $parcela['id_parcela_contrato'] = $parcela['nr_boleto'];
                                $pagamento['nosso_numero'] = $parcela['nr_boleto'];
                            }
                            continue;
                        } elseif ($this->tamanho_linha == 400) {
                            $juros = $juros_mora;
                        }

                        if (!Helper::dateIsValid($data_credito)) {
                            $dados[0]['erro'] = 'Tit. no pago - Sem Data Crdito';
                            $this->parcelasErradas[] = array_merge($pagamento, $dados[0]);
                            continue;
                        }

                        if (!Helper::dateIsValid($data_pagamento)) {
                            $dados[0]['erro'] = 'Tit. no pago - Sem Data Pagamento';
                            $this->parcelasErradas[] = array_merge($pagamento, $dados[0]);
                            continue;
                        }

                        //setando os dados na parcela que foi recebida baixa
                        $parcela->pago = '1';

                        $parcela->tp_baixa = 'B';
                        $parcela->fm_baixa = 'BL';

                        //se o valor pago no boleto for diferente do valor registrado da parcela no banco (no caso de 2a via)
                        if ($valorParcela < $valor_boleto) {
                            $parcela->vl_total_pago = $valor_boleto;
                            $juros = ($valor_boleto - $valorParcela);
                        } else {
                            $parcela->vl_total_pago = $valor_boleto;
                        }

                        //Calculo de multa para incluir no banco de dados, caso juros serja igual o menor que a multa nao deduz
                        if ($juros > 0) {
                            $percentagemMulta = Banco::getMultaByContrato($parcela->id_contrato);
                            $multa = (float)$percentagemMulta * $parcela->vl_parcela * 0.01;
                            if ($juros > $multa) {
                                $juros = $juros - $multa;
                                $parcela->vl_multa = $multa;
                            }
                        }

                        //Teste para registrar o banco na nova coluna de parcela (saber que banco registrou a entrada)
                        if ($dados[0]['id_banco'])
                            $parcela->id_banco = $dados[0]['id_banco'];
                        else
                            $parcela->id_banco = $banco['id'];

                        //Salva o txid para gerar o qrcode
                        if ($pagamento['banco'] == '748' && isset($this->pagamentos[$k + 1]['txid'])) {
                            $parcela->txid = $this->pagamentos[$k + 1]['txid'];
                        }

                        $parcela->vl_juros = round($juros, 2);
                        $parcela->acrescimo = 0;
                        $parcela->dt_pagamento = $data_pagamento;
                        $parcela->dt_credito = $data_credito;
                        $parcela->log = (Login::getUsuario()->getLogin() . ' - ' . date('d/m/Y H:m:s') . ' - u');
                        //persiste a parcela no banco de dados
                        $salvo = $parcela->save();

                        if ($banco['cd_banco'] == '748' || $banco['cd_banco'] == '741') {
                            $pagamento['nosso_numero'] = $parcela['nr_boleto'];
                        }


                        //se salvou com sucesso adiciona no array de parcelas corretas
                        if ($salvo) {
                            if (isset($pagamento['verificar']) && $pagamento['verificar'] == true) {
                                if ($_POST['cd_banco'] == Banco::BANCO_DO_BRASIL) {
                                    $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 7, 10);
                                } elseif ($_POST['cd_banco'] == Banco::BANCO_CAIXA_ECONOMICA) {
                                    $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 9, 10);
                                } elseif ($_POST['cd_banco'] == Banco::BANCO_BRADESCO) {
                                    $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 8, 11);
                                } elseif ($_POST['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO) {
                                    $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 8, 11);
                                } elseif ($_POST['cd_banco'] == Banco::BANCO_UNIPRIME) {
                                    $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 8, 11);
                                } elseif ($_POST['cd_banco'] == Banco::BANCO_SANTANDER) {
                                    $pagamento['nosso_numero'] = (substr($pagamento['numero_documento'], 5, 10) > 0) ? $pagamento['numero_documento'] : $pagamento['nosso_numero'];
                                } elseif ($_POST['cd_banco'] == Banco::BANCO_ITAU) {
                                    //Condicao introduzida para verificar nosso numero com digito verificador.
                                    if ($migradas == True) {
                                        $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 3, 9);
                                    } else {
                                        $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 3, 8);
                                    }
                                } elseif ($_POST['cd_banco'] == Banco::BANCO_SICREDI) {
                                    $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 2, 6);
                                } elseif ($_POST['cd_banco'] == Banco::BANCO_SICOOB || $_POST['cd_banco'] == Banco::BANCO_NORDESTE) {
                                    $pagamento['nosso_numero'] = (int)$pagamento['nosso_numero'];
                                } elseif ($_POST['cd_banco'] == Banco::BANCO_BANPARA) {
                                    $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 9, 10);
                                } elseif ($_POST['cd_banco'] == Banco::BANCO_SAFRA) {
                                    $pagamento['nosso_numero'] = (int)$pagamento['nosso_numero'];
                                }  elseif ($_POST['cd_banco'] == "085") {
                                    $pagamento["nosso_numero"] = str_replace(' ', '', $pagamento["nosso_numero"]);
                                    $pagamento["nosso_numero"] = substr($pagamento["nosso_numero"], 2, 19);
                                    $pagamento["nosso_numero"] = "00{$pagamento["nosso_numero"]}";
                                }elseif ($_POST['cd_banco'] == "136") {
                                    $pagamento['nosso_numero'] = trim($pagamento['nosso_numero']);
                                }else {
                                    $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 7, 11);
                                }
                               
                                if ($_POST['exibirTodos'] == '0' && $pagamento['cod_movimento'] == '02')
                                    continue;
                                $this->parcelasVerificar[] = array_merge($pagamento, $dados[0]);
                            } else {
                                $this->parcelasCorretas[] = array_merge($dados[0], $parcela->toArray());
                            }
                        } else {
                            if ($_POST['exibirTodos'] == '0' && $pagamento['cod_movimento'] == '02')
                                continue;
                            if ($_POST['cd_banco'] == Banco::BANCO_DO_BRASIL) {
                                $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 7, 10);
                            } elseif ($_POST['cd_banco'] == Banco::BANCO_CAIXA_ECONOMICA) {
                                $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 9, 10);
                            } elseif ($_POST['cd_banco'] == Banco::BANCO_BRADESCO) {
                                $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 8, 11);
                            } elseif ($_POST['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO) {
                                $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 8, 11);
                            } elseif ($_POST['cd_banco'] == Banco::BANCO_UNIPRIME) {
                                $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 8, 11);
                            } elseif ($_POST['cd_banco'] == Banco::BANCO_SANTANDER) {
                                $pagamento['nosso_numero'] = (substr($pagamento['numero_documento'], 5, 10) > 0) ? $pagamento['numero_documento'] : $pagamento['nosso_numero'];;
                            } elseif ($_POST['cd_banco'] == Banco::BANCO_ITAU) {
                                //Condicao introduzida para verificar nosso numero com digito verificador.
                                if ($migradas == True) {
                                    $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 3, 9);
                                } else {
                                    $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 3, 8);
                                }
                            } elseif ($_POST['cd_banco'] == Banco::BANCO_SICREDI) {
                                $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 2, 6);
                            } elseif ($_POST['cd_banco'] == Banco::BANCO_SICOOB || $_POST['cd_banco'] == Banco::BANCO_NORDESTE) {
                                $pagamento['nosso_numero'] = (int)$pagamento['nosso_numero'];
                            } elseif ($_POST['cd_banco'] == Banco::BANCO_BANPARA) {
                                $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 9, 10);
                            } elseif ($_POST['cd_banco'] == Banco::BANCO_SAFRA) {
                                $pagamento['nosso_numero'] = (int)$pagamento['nosso_numero'];
                            }
                            elseif ($_POST['cd_banco'] == "085") {
                                $pagamento["nosso_numero"] = str_replace(' ', '', $pagamento["nosso_numero"]);
                                $pagamento["nosso_numero"] = substr($pagamento["nosso_numero"], 2, 19);
                                $pagamento["nosso_numero"] = "00{$pagamento["nosso_numero"]}";
                            }
                            elseif ($_POST['cd_banco'] == "136") {
                                $pagamento['nosso_numero'] = trim($pagamento['nosso_numero']);
                            } else {
                                $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 7, 11);
                            }
                            
                            $pagamento['vl_parcela'] = $valor_boleto;
                            $this->parcelasErradas[] = array_merge($pagamento, $dados[0]);
                        }

                    } elseif ($this->tamanho_linha == 400 && $comRegistro && $pagamento['cod_movimento'] != '06') {
                        continue;
                    } else {
                        if ($_POST['exibirTodos'] == '0' && $pagamento['cod_movimento'] == '02')
                            continue;
                        if ($_POST['cd_banco'] == Banco::BANCO_DO_BRASIL) {
                            $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 7, 10);
                        } elseif ($_POST['cd_banco'] == Banco::BANCO_CAIXA_ECONOMICA) {
                            $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 9, 10);
                        } elseif ($_POST['cd_banco'] == Banco::BANCO_SANTANDER) {
                            $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 2, 10);
                        } elseif ($_POST['cd_banco'] == Banco::BANCO_BRADESCO) {
                            $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 8, 11);
                        } elseif ($_POST['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO) {
                            $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 8, 11);
                        } elseif ($_POST['cd_banco'] == Banco::BANCO_UNIPRIME) {
                            $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 8, 11);
                        } elseif ($_POST['cd_banco'] == Banco::BANCO_ITAU) {
                            //Condicao introduzida para verificar nosso numero com digito verificador.
                            if ($migradas) {
                                $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 3, 9);
                            } else {
                                $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 3, 8);
                            }
                        } elseif ($_POST['cd_banco'] == Banco::BANCO_NORDESTE) {
                            $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 0, 11);
                        } elseif ($_POST['cd_banco'] == Banco::BANCO_SICREDI) {
                            $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 2, 6);
                        } elseif ($_POST['cd_banco'] == Banco::BANCO_SICOOB || $_POST['cd_banco'] == Banco::BANCO_NORDESTE) {
                            $pagamento['nosso_numero'] = trim($pagamento['nosso_numero']);
                        } elseif ($_POST['cd_banco'] == Banco::BANCO_BANPARA) {
                            $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 9, 10);
                        } elseif ($_POST['cd_banco'] == Banco::BANCO_SAFRA) {
                            $pagamento['nosso_numero'] = trim($pagamento['nosso_numero']);
                        } 
                        elseif ($_POST['cd_banco'] == "085") {
                            $pagamento["nosso_numero"] = str_replace(' ', '', $pagamento["nosso_numero"]);
                            $pagamento["nosso_numero"] = substr($pagamento["nosso_numero"], 2, 19);
                            $pagamento["nosso_numero"] = "00{$pagamento["nosso_numero"]}";
                        }
                        elseif ($_POST['cd_banco'] == "136") {
                            $pagamento['nosso_numero'] = trim($pagamento['nosso_numero']);
                        }else {
                            $pagamento['nosso_numero'] = substr($pagamento['nosso_numero'], 7, 11);
                        }
                        $pagamento['vl_parcela'] = $valor_boleto;

                        $this->parcelasErradas[] = $pagamento;
                    }
                }
            }

            // Resultado do processamento
            function ordenarParcelas($a, $b)
            {
                if ($a['contrato']->nm_empreendimento == $b['contrato']->nm_empreendimento) {
                    return 0;
                } else {
                    return ($a['contrato']->nm_empreendimento < $b['contrato']->nm_empreendimento) ? 1 : -1;
                }
            }

            if ($this->parcelasCorretas && sizeof($this->parcelasCorretas) > 0) {
                usort($this->parcelasCorretas, "ordenarParcelas");
            }

            if ($this->parcelasErradas && sizeof($this->parcelasErradas) > 0) {
                usort($this->parcelasErradas, "ordenarParcelas");
            }

            if ($this->parcelasVerificar && sizeof($this->parcelasVerificar) > 0) {
                usort($this->parcelasVerificar, "ordenarParcelas");
            }

            $resultado['sucessos'] = $this->parcelasCorretas;
            $resultado['erros'] = $this->parcelasErradas;
            $resultado['verificar'] = $this->parcelasVerificar;
            $resultado['linhasErradas'] = $this->linhasErradas;
            $resultado['codigosMovimento'] = $this->codigosMovimento;
        }

        return $resultado;
    }

    /**
     * Carrega o arquivo de retorno bancario e processa as linhas obtidas.
     *
     * @param void
     * @return void
     * @author Pablo de Oliveira Lima.
     */
    public function processarArquivo()
    {
        $linhas = file($this->arquivo, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        foreach ($linhas as $key => $value) {
            $linhas[$key] = utf8_decode($linhas[$key]);
        }

        foreach ($linhas as $numLn => $linha) {
            if ($_POST['cd_banco'] == Banco::BANCO_ITAU)
                $linhaProcessada = $this->processarLinha($numLn, $linha, $migradas = true);
            else
                $linhaProcessada = $this->processarLinha($numLn, $linha);

            if ($linhaProcessada && $this->tamanho_linha == 400) {
                if ($linhaProcessada['linha']['sequencial'] && $linhaProcessada['tipo'] == '1') {
                    $this->pagamentos[] = $linhaProcessada['linha'];
                }
            } elseif ($linhaProcessada && $this->tamanho_linha == 240) {
                // Verifica se eh registro de detalhe de pagamento
                if ($linhaProcessada['linha']['registro'] == '3') {
                    // Verifica se o tipo de linha tem segmento U,
                    // e caso tenha pega o segmento T anteriormente adicionado e concatena
                    if ($linhaProcessada['linha']['segmento'] == "U") {
                        $parte1 = array_pop($this->pagamentos); // retirando segmento T da lista
                        $parte2 = $linhaProcessada['linha'];
                        $this->pagamentos[] = array_merge($parte1, $parte2);
                    } else {
                        $this->pagamentos[] = $linhaProcessada['linha'];
                    }
                }
            }
        }
    }

    /**
     * Processa uma linha obtida do arquivo de retorno bancario.
     *
     * @param int $numLn Numero da linha no arquivo.
     * @param string $linha Linha obtida do arquivo.
     * @return array $vlinha  Retorna os dados da linha como array
     * @author Pablo de Oliveira Lima, Manoel Campos da Silva Filho
     */

    protected function processarLinha($numLn, $linha, $migradas = false)
    {
        //total de caracteres das linhas do arquivo
        $tamLinha = $this->tamanho_linha;
        //total de caracteres das linhas do arquivo e o +2 e utilizado para contar o \r\n no final da linha
        if (strlen($linha) != $tamLinha || (trim($linha) == "")) {
            $this->linhasErradas[] = $linha;
            return false;
        }
        //? adicionado um espaco vazio no inicio_linha para que
        //possamos trabalhar com indices iniciando_1, no lugar_zero,
        //e assim, ter os valores_posicao_campos exatamente
        //como no manual CNAB400 / CNAB240
        $linha = " $linha";
        if ($this->tamanho_linha == 400) {
            $tipoLn = substr($linha, 1, 1);

            if ($tipoLn == RetornoBoleto::HEADER_ARQUIVO) {
                $vlinha = array('tipo' => RetornoBoleto::HEADER_ARQUIVO,
                    'linha' => $this->processarLinhaHeader($linha));
                $this->header_arquivo = $vlinha['linha'];
            } else if ($tipoLn == RetornoBoleto::DETALHE_400) {
                $vlinha = array('tipo' => RetornoBoleto::DETALHE_400,
                    'linha' => $this->processarLinhaDetalhe($linha, $migradas));
            } else if ($tipoLn == RetornoBoleto::TRAILER_ARQUIVO) {
                $vlinha = array('tipo' => RetornoBoleto::TRAILER_ARQUIVO,
                    'linha' => $this->processarLinhaTrailer($linha));
                $this->trailer_arquivo = $vlinha['linha'];
            }
        } elseif ($this->tamanho_linha == 240) {
            $tipoLn = substr($linha, 8, 1);

            if ($tipoLn == RetornoBoleto::HEADER_ARQUIVO) {
                $vlinha = array('tipo' => RetornoBoleto::HEADER_ARQUIVO,
                    'linha' => $this->processarLinhaHeader($linha));
                $this->header_arquivo = $vlinha['linha'];
            } else if ($tipoLn == RetornoBoleto::HEADER_LOTE) {
                $vlinha = array('tipo' => RetornoBoleto::HEADER_LOTE,
                    'linha' => $this->processarLinhaHeaderLote($linha));
                $this->header_lote = $vlinha['linha'];
            } else if ($tipoLn == RetornoBoleto::DETALHE_240) {
                $vlinha = array('tipo' => RetornoBoleto::DETALHE_240,
                    'linha' => $this->processarLinhaDetalhe($linha));
            } else if ($tipoLn == RetornoBoleto::TRAILER_ARQUIVO) {
                $vlinha = array('tipo' => RetornoBoleto::TRAILER_ARQUIVO,
                    'linha' => $this->processarLinhaTrailer($linha));
                $this->trailer_arquivo = $vlinha['linha'];
            } else if ($tipoLn == RetornoBoleto::TRAILER_LOTE) {
                $vlinha = array('tipo' => RetornoBoleto::TRAILER_LOTE,
                    'linha' => $this->processarLinhaTrailerLote($linha));
                $this->trailer_lote = $vlinha['linha'];
            }
        }

        return $vlinha;
    }

    protected function processarLinhaHeader($linha)
    {
        $vlinha = array();
        if ($this->tamanho_linha == 400) {
            $this->banco = substr($linha, 77, 3);
            //X = ALFANUMERICO 9 = NUMERICO V = VIRGULA DECIMAL ASSUMIDA
            $vlinha["id_registro"] = substr($linha, 1, 1); //9 Identificacao do Registro Header: ?0?
            $vlinha["tipo_operacao"] = substr($linha, 2, 1); //9 Tipo de Operacao: ?2?
            $vlinha["id_tipo_operacao"] = substr($linha, 3, 7); //X Identificacao Tipo de Operacao ?RETORNO?
            $vlinha["id_tipo_servico"] = substr($linha, 10, 2); //9 Identificacao do Tipo de Servico: ?01?
            $vlinha["tipo_servico"] = substr($linha, 12, 8); //X Identificacao por Extenso do Tipo de Servico: ?COBRANCA?
            $vlinha["codigo_empresa"] = substr($linha, 27, 20);
            $vlinha["razao_social"] = substr($linha, 47, 30);
            $vlinha["cod_banco"] = substr($linha, 77, 3);
            $vlinha["nome_banco"] = substr($linha, 80, 15);
            $vlinha["data_gravacao"] = $this->formataData(substr($linha, 95, 6));
            $vlinha["densidade_gravacao"] = substr($linha, 101, 8);
            $vlinha["aviso_bancario"] = substr($linha, 109, 5);
            $vlinha["complemento2"] = substr($linha, 114, 266);
            $vlinha["data_credito"] = $this->formataData(substr($linha, 380, 6));
            $vlinha["complemento3"] = substr($linha, 386, 9);
            $vlinha["sequencial_reg"] = substr($linha, 395, 6);
        } elseif ($this->tamanho_linha == 240) {
            $this->banco = substr($linha, 1, 3);
            $vlinha["banco"] = substr($linha, 1, 3); //NUMERICO //Codigo do Banco na Compensacao
            $vlinha["lote"] = substr($linha, 4, 4); //num - default 0000 //Lote de Servi?o
            $vlinha["registro"] = substr($linha, 8, 1); //num - default 0 //Tipo de Registro
            $vlinha["cnab1"] = substr($linha, 9, 9); //BRANCOS //Uso Exclusivo FEBRABAN / CNAB
            $vlinha["tipo_inscricao_empresa"] = substr($linha, 18, 1); //num - 1-CPF, 2-CGC //Tipo de Inscricao da Empresa
            $vlinha["num_inscricao_empresa"] = substr($linha, 19, 14); //numerico  //Numero de Inscricao da Empresa
            $vlinha["cod_convenio"] = substr($linha, 33, 20); //alfanumerico  //Codigo do Convenio no Banco

            $vlinha["agencia"] = substr($linha, 53, 5); //numerico //Agencia Mantenedora da Conta
            $vlinha["dv_agencia"] = substr($linha, 58, 1); //alfanumerico //DV da Agencia
            $vlinha["conta_corrente"] = substr($linha, 59, 12); //numerico //Numero da Conta Corrente
            $vlinha["dv_conta"] = substr($linha, 71, 1); //alfanumerico  //DV da Conta Corrent

            if ($this->banco == '341') {
                $vlinha["dv_agencia"] = substr($linha, 72, 1); //alfanumerico //DV da Agencia
                $vlinha["dv_conta"] = substr($linha, 72, 1); //alfanumerico  //DV da Conta Corrent
            }

            if ($this->banco == '033') {
                $vlinha["agencia"] = substr($linha, 33, 4); //numerico //Agencia Mantenedora da Conta
                $vlinha["dv_agencia"] = substr($linha, 37, 1); //alfanumerico //DV da Agencia
                $vlinha["conta_corrente"] = substr($linha, 38, 9); //numerico //Numero da Conta Corrente
                $vlinha["dv_conta"] = substr($linha, 47, 1); //alfanumerico  //DV da Conta Corrent
            } elseif ($this->banco == '104') {
                $vlinha["conta_corrente"] = ltrim(rtrim(substr($linha, 59, 7), '0'), '0'); //numerico //Numero da Conta Corrente

                //Caso o cedente seja maior que 7 digitos, busca os 7 digitos
                if (strlen($vlinha["conta_corrente"]) > 6) {
                    $vlinha["conta_corrente"] = substr($linha, 59, 7); //numerico //Numero da Conta Corrente
                }

                $vlinha["dv_conta"] = '0'; //alfanumerico  //DV da Conta Corrent
            } elseif ($this->banco == '741') {
                $vlinha["agencia"] = substr($linha, 56, 4); //numerico //Agencia Mantenedora da Conta
                $vlinha["dv_agencia"] = substr($linha, 60, 1); //alfanumerico //DV da Agencia
                $vlinha["conta_corrente"] = substr($linha, 61, 6); //numerico //Numero da Conta Corrente
            }
            elseif ($this->banco == '136') {
                $vlinha["agencia"] = substr($linha, 53, 5); //numerico //Agencia Mantenedora da Conta
                $vlinha["dv_agencia"] = substr($linha, 58, 1); //alfanumerico //DV da Agencia
                $vlinha["conta_corrente"] = substr($linha, 59, 13); //numerico //Numero da Conta Corrente
                $vlinha["dv_conta"] = substr($linha, 72, 1); //alfanumerico  //DV da Conta Corrent
            }
            $vlinha["dv_ag_conta"] = substr($linha, 72, 1); //alfanumerico
            $vlinha["nome_empresa"] = substr($linha, 73, 30); //alfanumerico
            $vlinha["nome_banco"] = substr($linha, 103, 30); //alfanumerico
            $vlinha["uso_febraban_cnab2"] = substr($linha, 133, 10); //brancos //Uso Exclusivo FEBRABAN / CNAB
            $vlinha["cod_arq"] = substr($linha, 143, 1); //num - 1-REM E 2-RET ?? //Codigo do arquivo de remessa/retorno
            $vlinha["data_geracao_arq"] = substr($linha, 144, 8); //num - formato ddmmaaaa
            $vlinha["hora_geracao_arq"] = substr($linha, 152, 6); //num - formato hhmmss
            $vlinha["sequencia"] = substr($linha, 158, 6); //numerico //Numero Sequencial do Arquivo
            $vlinha["versao_layout_arq"] = substr($linha, 164, 3); //num 084 //Num da Versao do Layout do Arquivo
            $vlinha["densidade"] = substr($linha, 167, 5); //numerico //Densidade de Gravacao do Arquivo
            $vlinha["reservado_banco"] = substr($linha, 172, 20); //alfanumerico //Para Uso Reservado do Banco
            $vlinha["reservado_empresa"] = substr($linha, 192, 20); //alfanumerico //Para Uso Reservado da Empresa
            $vlinha["uso_febraban_cnab3"] = substr($linha, 212, 29); //brancos //Uso Exclusivo FEBRABAN / CNAB
        }

        return $vlinha;
    }

    protected function processarLinhaHeaderLote($linha)
    {
        $vlinha = array();
        $vlinha["banco"] = substr($linha, 1, 3); //numerico //Codigo do Banco na Compensacao
        $vlinha["lote"] = substr($linha, 4, 4); //numerico //Lote de Servico
        $vlinha["registro"] = substr($linha, 8, 1); //num - default 1 //Tipo de Registro
        $vlinha["operacao"] = substr($linha, 9, 1); //alfanumerico - default C //Tipo da Operacao
        $vlinha["servico"] = substr($linha, 10, 2); //num  //Tipo do Servico
        $vlinha["forma_lancamento"] = substr($linha, 12, 2); //num //Forma de Lancamento
        $vlinha["layout_lote"] = substr($linha, 14, 3); //num - default '030' //No da Versao do Layout do Lote
        $vlinha["cnab1"] = substr($linha, 17, 1); //alfa - default brancos  //Uso Exclusivo da FEBRABAN/CNAB

        $vlinha["tipo_inscricao_empresa"] = substr($linha, 18, 1); //num - 1-CPF, 2-CGC //Tipo de Inscricao da Empresa
        $vlinha["num_inscricao_empresa"] = substr($linha, 20, 14); //numerico //Numero de Inscricao da Empresa
        $vlinha["cod_convenio"] = substr($linha, 34, 20); //alfanumerico //Codigo do Convenio no Banco

        $vlinha["agencia"] = substr($linha, 54, 5); //numerico //Agencia Mantenedora da Conta
        $vlinha["dv_agencia"] = substr($linha, 59, 1); //alfanumerico //DV da Agencia Mantenedora da Conta
        $vlinha["conta_corrente"] = substr($linha, 60, 12); //numerico
        $vlinha["dv_conta"] = substr($linha, 72, 1); //alfanumerico
        $vlinha["dv_ag_conta"] = substr($linha, 73, 1); //alfanumerico //Digito Verificador da Ag/Conta
        $vlinha["nome_empresa"] = substr($linha, 74, 30); //alfanumerico
        $vlinha["mensagem1"] = substr($linha, 104, 40); //alfanumerico

        $vlinha["logradouro_empresa"] = substr($linha, 144, 30); //alfa //Logradouro da Empresa - Nome da Rua, Av, Pa, Etc
        $vlinha["numero_empresa"] = substr($linha, 174, 5); //num //Numero do endereco da empresa
        $vlinha["complemento_empresa"] = substr($linha, 179, 15); //alfa //Complemento - Casa, Apto, Sala, Etc
        $vlinha["cidade_empresa"] = substr($linha, 194, 20); //alfa //Cidade da Empresa
        $vlinha["cep_empresa"] = substr($linha, 214, 5); //num //5 primeiros digitos do CEP da Empresa
        $vlinha["complemento_cep_empresa"] = substr($linha, 219, 3); //alfa //3 Ultimos digitos do CEP da empresa
        $vlinha["estado"] = substr($linha, 222, 2); //  alfa  //Sigla do Estado
        $vlinha["cnab"] = substr($linha, 224, 8); // alfa - default brancos //Uso Exclusivo da FEBRABAN/CNAB
        $vlinha["ocorrencias"] = substr($linha, 232, 10); //alfa //Codigo das Ocorrencias p/ Retorno
        return $vlinha;
    }

    protected function processarLinhaDetalhe($linha, $migradas = false)
    {
        $vlinha = array();

        if ($this->tamanho_linha == 400) {

            // Se o valor pago for zero, nao deve baixar.
            $valorPago = $this->formataNumero(substr($linha, 254, 13));
            if ((float)$valorPago == 0) {
                return null;
            }

            //X = ALFANUM?RICO 9 = NUMERICO V = VIRGULA DECIMAL ASSUMIDA
            $vlinha["id_registro"] = substr($linha, 1, 1); //9  Identificacao do Registro Detalhe: 1 (um)
            $vlinha["tipo_insc_empresa"] = substr($linha, 2, 2);
            $vlinha["nr_insc_empresa"] = substr($linha, 4, 14);
            $vlinha["zeros1"] = substr($linha, 18, 3);
            $vlinha["cod_cedente"] = substr($linha, 21, 17);
            $vlinha["contr_participante"] = substr($linha, 38, 25);
            $vlinha["zeros2"] = substr($linha, 63, 8);
            $vlinha["titulo_banco"] = substr($linha, 71, 12);
            $vlinha["zeros3"] = substr($linha, 83, 10);
            $vlinha["zeros4"] = substr($linha, 93, 12);
            $vlinha["rateio_credito"] = substr($linha, 105, 1);
            $vlinha["zeros5"] = substr($linha, 106, 2);
            $vlinha["carteira"] = substr($linha, 108, 1);
            $vlinha["cod_movimento"] = substr($linha, 109, 2); //Codigo do movimento
            $vlinha["data_ocorrencia"] = $this->formataData(substr($linha, 111, 6));
            $vlinha["numero_documento"] = substr($linha, 117, 10);
            $vlinha["titulo_banco2"] = substr($linha, 127, 20);

            $vlinha["valor_pago"] = $this->formataNumero(substr($linha, 254, 13));

            $calcJuros = false;

            if ((float)substr($linha, 147, 6) == 0) {
                $cdBoleto = substr($vlinha['titulo_banco'], 1, 10);
                if ($migradas == true)
                    $boleto = $this->tb_parcela->fetchAll("id_parcela_contrato_antigo = '" . $cdBoleto . "' ")->toArray();
                else
                    $boleto = $this->tb_parcela->fetchAll("id_parcela_contrato = '" . $cdBoleto . "' ")->toArray();
                $zendDate = new Zend_Date($boleto[0]['dt_parcela']);
                $vlinha["data_vencimento"] = $zendDate->get(Zend_Date::DATE_MEDIUM);

                $valorParcela = $boleto[0]['vl_parcela'];
                if ((float)$vlinha["valor_pago"] > (float)$valorParcela) {
                    $calcJuros = true;
                    $vlinha["juros_mora"] = (string)((float)$vlinha["valor_pago"] - (float)$valorParcela);
                }

            } else {
                $vlinha["data_vencimento"] = $this->formataData(substr($linha, 147, 6));
            }


            $vlinha["valor_titulo"] = $this->formataNumero(substr($linha, 153, 13));
            $vlinha["codigo_banco_cobr"] = substr($linha, 166, 3);
            $vlinha["agencia_banco_cobr"] = substr($linha, 169, 5);
            $vlinha["especie_titulo"] = substr($linha, 174, 2);
            $vlinha["valor_tarifa"] = $this->formataNumero(substr($linha, 176, 13));
            $vlinha["outras_despesas"] = $this->formataNumero(substr($linha, 189, 13));
            $vlinha["juros_operacao"] = $this->formataNumero(substr($linha, 202, 13));
            $vlinha["iof_devido"] = $this->formataNumero(substr($linha, 215, 13));
            $vlinha["valor_abatimento"] = $this->formataNumero(substr($linha, 228, 13));
            $vlinha["desconto_concedido"] = $this->formataNumero(substr($linha, 241, 13));

            if (!$calcJuros) {
                $vlinha["juros_mora"] = $this->formataNumero(substr($linha, 267, 13));
            }

            $vlinha["outros_creditos"] = $this->formataNumero(substr($linha, 280, 13));
            $vlinha["brancos1"] = substr($linha, 293, 2);
            $vlinha["motivo_ocorrencia"] = substr($linha, 295, 1);
            $vlinha["data_credito"] = $this->formataData(substr($linha, 296, 6));
            $vlinha["origem_pagamento"] = substr($linha, 302, 3);
            $vlinha["brancos2"] = substr($linha, 305, 10);
            $vlinha["cheque_bradesco"] = substr($linha, 315, 4);
            $vlinha["motivo_ocorrencia2"] = substr($linha, 319, 10);
            $vlinha["brancos3"] = substr($linha, 329, 40);
            $vlinha["cartorio"] = substr($linha, 369, 2);
            $vlinha["protocolo"] = substr($linha, 371, 10);
            $vlinha["brancos4"] = substr($linha, 381, 10);
            $vlinha["sequencial"] = substr($linha, 395, 6);
        } elseif ($this->tamanho_linha == 240) {

            $vlinha["banco"] = substr($linha, 1, 3); //   Num //Codigo no Banco da Compensacao
            $vlinha["lote"] = substr($linha, 4, 4); //   Num //Lote de Servico
            $vlinha["registro"] = substr($linha, 8, 1); //   Num  default '3' //Tipo de Registro
            $vlinha["num_registro_lote"] = substr($linha, 9, 5); //   Num  //No Sequencial do Registro no Lote
            $vlinha["segmento"] = substr($linha, 14, 1); //   Alfa  default 'J' //Codigo de Segmento no Reg. Detalhe
            $vlinha["tipo_movimento"] = substr($linha, 15, 1); //   Num //Tipo de Movimento
            $vlinha["cod_movimento"] = substr($linha, 16, 2); //   Num  //Codigo da Instrucao p/ Movimento

            switch ($vlinha["segmento"]) {
                case 'J':
                    $vlinha["cod_barras"] = substr($linha, 18, 44); //   Num
                    $vlinha["nome_cedente"] = substr($linha, 62, 30); //   Alfa
                    $vlinha["data_vencimento"] = substr($linha, 92, 8); //   Num  //Data do Vencimento (Nominal)
                    $vlinha["valor_titulo"] = substr($linha, 100, 15); //   Num, 2 casas decimais //Valor do Titulo (Nominal)
                    $vlinha["desconto"] = substr($linha, 115, 15); //   Num, 2 casas decimais //Valor do Desconto + Abatimento
                    $vlinha["acrescimos"] = substr($linha, 130, 15); //   Num, 2 casas decimais //Valor da Mora + Multa
                    $vlinha["data_pagamento"] = substr($linha, 145, 8); //   Num
                    $vlinha["valor_pagamento"] = substr($linha, 153, 15); //   Num, 2 casas decimais
                    $vlinha["quantidade_moeda"] = substr($linha, 168, 15); //   Num, 5 casas decimais
                    $vlinha["referencia_sacado"] = substr($linha, 183, 20); //   Alfa //Num. do Documento Atribuido pela Empresa
                    $vlinha["nosso_numero"] = substr($linha, 203, 20); //   Alfa //Num. do Documento Atribu?do pelo Banco
                    $vlinha["cod_moeda"] = substr($linha, 223, 2); //   Num
                    $vlinha["cnab"] = substr($linha, 225, 6); //   Alfa - default Brancos //Uso Exclusivo FEBRABAN/CNAB
                    $vlinha["ocorrencias"] = substr($linha, 231, 10); //   Alfa //Codigos das Ocorrencias p/ Retorno
                    break;
                case 'T':
                    if ($this->banco == '033') { //Santander
                        $vlinha["agencia_conta"] = substr($linha, 18, 4); // Num
                        $vlinha["dv_agencia"] = substr($linha, 22, 1); // Num
                        $vlinha["numero_conta"] = substr($linha, 23, 9); // Num
                        $vlinha["dv_conta"] = substr($linha, 32, 1); // Num
                        $vlinha["dv_agencia_conta"] = substr($linha, 33, 8); // Num
                        $vlinha["nosso_numero"] = substr($linha, 41, 12); // Alfa
                        $vlinha["carteira"] = substr($linha, 54, 1); // Num
                        $vlinha["numero_documento"] = substr($linha, 55, 15); // Alfa
                        $vlinha["vencimento"] = substr($linha, 70, 8); // Num
                        $vlinha["valor_titulo"] = substr($linha, 78, 15); // Num
                        $vlinha["numero_banco"] = substr($linha, 93, 3); // Num
                        $vlinha["agencia_cobradora"] = substr($linha, 96, 4); // Num
                        $vlinha["dv_agencia_cobrad"] = substr($linha, 100, 1); // Num
                        $vlinha["titulo_empresa"] = substr($linha, 101, 25); // Alfa
                        $vlinha["cod_moeda"] = substr($linha, 126, 2); // Num
                        $vlinha["inscricao_sacado"] = substr($linha, 128, 1); // Num
                        $vlinha["inscricao_sacado"] = substr($linha, 129, 15); // Num
                        $vlinha["sacado"] = substr($linha, 144, 40); // Alfa
                        $vlinha["numero_contrato"] = substr($linha, 184, 10); // Num
                        $vlinha["valor_tarifa"] = substr($linha, 194, 15); // Num
                        $vlinha["ocorrencias"] = substr($linha, 209, 10); // Alfa
                        $vlinha["cnab"] = substr($linha, 219, 22); // Alfa
                    } else {
                        $vlinha["agencia_conta"] = substr($linha, 18, 5); // Num
                        $vlinha["dv_agencia"] = substr($linha, 23, 1); // Num
                        $vlinha["numero_conta"] = substr($linha, 24, 12); // Num
                        $vlinha["dv_conta"] = substr($linha, 36, 1); // Num
                        $vlinha["dv_agencia_conta"] = substr($linha, 37, 1); // Num
                        if ($this->banco == '756') { //Sicoob
                            $vlinha["nosso_numero"] = substr($linha, 59, 15); // Alfa
                        } elseif ($this->banco == '085') { //Banco Ailos
                            $vlinha["nosso_numero"] = substr($linha, 38, 20); // Alfa
                        } elseif ($this->banco == '004') { //Banco Nordeste
                            $vlinha["nosso_numero"] = substr($linha, 38, 7); // Alfa
                        } elseif ($this->banco == '136') { //Banco Nordeste
                            $vlinha["nosso_numero"] = substr($linha, 39, 19); // Alfa
                        } elseif ($this->banco == '099') { //Banco Uniprime
                            $vlinha["nosso_numero"] = substr($linha, 38, 20); // Alfa
                        }else { //Outros Bancos
                            $vlinha["nosso_numero"] = substr($linha, 38, 20); // Alfa
                        }
                        $vlinha["carteira"] = substr($linha, 58, 1); // Num
                        $vlinha["numero_documento"] = substr($linha, 59, 15); // Alfa
                        $vlinha["vencimento"] = substr($linha, 74, 8); // Num
                        $vlinha["valor_titulo"] = substr($linha, 82, 15); // Num
                        $vlinha["numero_banco"] = substr($linha, 97, 3); // Num
                        $vlinha["agencia_cobradora"] = substr($linha, 100, 5); // Num
                        $vlinha["dv_agencia_cobrad"] = substr($linha, 105, 1); // Num
                        $vlinha["titulo_empresa"] = substr($linha, 106, 25); // Alfa
                        $vlinha["cod_moeda"] = substr($linha, 131, 2); // Num
                        $vlinha["inscricao_sacado"] = substr($linha, 133, 16); // Num
                        $vlinha["sacado"] = substr($linha, 149, 40); // Alfa
                        $vlinha["numero_contrato"] = substr($linha, 189, 10); // Num
                        $vlinha["valor_tarifa"] = substr($linha, 199, 15); // Num
                        $vlinha["ocorrencias"] = substr($linha, 214, 10); // Alfa
                        $vlinha["cnab"] = substr($linha, 224, 17); // Alfa
                    }
                    break;
                case 'U':
                    $vlinha["acrescimos"] = substr($linha, 18, 15); // Num
                    $vlinha["desconto"] = substr($linha, 33, 15); // Num
                    $vlinha["abatimento"] = substr($linha, 48, 15); // Num
                    $vlinha["IOF"] = substr($linha, 63, 15); // Num
                    $vlinha["valor_pago"] = substr($linha, 78, 15); // Num
                    $vlinha["valor_liquido"] = substr($linha, 93, 15); // Num
                    $vlinha["outras_despesas"] = substr($linha, 108, 15); // Num
                    $vlinha["outros_creditos"] = substr($linha, 123, 15); // Num
                    $vlinha["data_ocorrencia"] = substr($linha, 138, 8); // Num
                    $vlinha["data_credito"] = substr($linha, 146, 8); // Num
                    if ($vlinha["data_credito"] == '00000000') {
                        $vlinha["data_credito"] = substr($linha, 138, 8); // Num
                    } else {
                        $vlinha["data_credito"] = substr($linha, 146, 8); // Num
                    }
                    $vlinha["ocorrencia_sacado"] = substr($linha, 154, 4); // Alfa
                    $vlinha["data_ocorrencia_sac"] = substr($linha, 158, 8); // Alfa
                    $vlinha["valor_ocorrencia_sac"] = substr($linha, 166, 15); // Num
                    $vlinha["compl_ocorrencia_sac"] = substr($linha, 181, 30); // Alfa
                    $vlinha["banco_compensacao"] = substr($linha, 211, 3); // Num
                    $vlinha["nosso_numero_comp"] = substr($linha, 214, 20); // Num
                    $vlinha["cnab"] = substr($linha, 234, 7); // Alfa
                    break;
                case 'Y':
                    $vlinha["txid"] = substr($linha, 159, 35); // Alfa
                    break;
            }
        }
        return $vlinha;
    }

    protected function processarLinhaTrailer($linha)
    {
        $vlinha = array();
        if ($this->tamanho_linha == 400) {
            //X = ALFANUMERICO 9 = NUMERICO V = VIRGULA DECIMAL ASSUMIDA
            $vlinha["id_registro"] = substr($linha, 1, 1); //9  Identificacao do Registro Trailer: ?9?
            $vlinha["2"] = substr($linha, 2, 1); //9  ?2?
            $vlinha["01"] = substr($linha, 3, 2); //9  ?01?
            $vlinha["001"] = substr($linha, 5, 3); //9  ?001?
            $vlinha["brancos"] = substr($linha, 8, 10); //X  Brancos
            $vlinha["cob_simples_qtd_titulos"] = substr($linha, 18, 8); //9  Cobranca Simples - quantidade de titulos
            $vlinha["cob_simples_vlr_total"] = $this->formataNumero(substr($linha, 26, 14)); //9  v99 Cobranca Simples - valor total
            $vlinha["cob_simples_num_aviso"] = substr($linha, 40, 8); //9  Cobranca Simples - Numero do aviso
            $vlinha["brancos2"] = substr($linha, 48, 10);
            $vlinha["cob_qtd_registros"] = substr($linha, 58, 5);
            $vlinha["cob_vlr_registros"] = $this->formataNumero(substr($linha, 63, 12));
            $vlinha["cob_vlr_registros2"] = $this->formataNumero(substr($linha, 75, 12));
            $vlinha["cob_qtd_registros2"] = substr($linha, 87, 5);
            $vlinha["cob_vlr_registros3"] = $this->formataNumero(substr($linha, 92, 12));
            $vlinha["cob_qtd_registros3"] = substr($linha, 104, 5);
            $vlinha["cob_vlr_registros4"] = $this->formataNumero(substr($linha, 109, 12));
            $vlinha["cob_qtd_registros4"] = substr($linha, 121, 5);
            $vlinha["cob_vlr_registros5"] = $this->formataNumero(substr($linha, 126, 12));
            $vlinha["cob_qtd_registros5"] = substr($linha, 138, 5);
            $vlinha["cob_vlr_registros6"] = $this->formataNumero(substr($linha, 143, 12));
            $vlinha["cob_qtd_registros6"] = substr($linha, 155, 5);
            $vlinha["cob_vlr_registros7"] = $this->formataNumero(substr($linha, 160, 12));
            $vlinha["cob_qtd_registros7"] = substr($linha, 172, 5);
            $vlinha["cob_vlr_registros8"] = $this->formataNumero(substr($linha, 177, 12));
            $vlinha["brancos3"] = substr($linha, 189, 174);
            $vlinha["valor_total_rateio"] = $this->formataNumero(substr($linha, 363, 15));
            $vlinha["qtd_rateios"] = $this->formataNumero(substr($linha, 378, 8));
            $vlinha["brancos4"] = substr($linha, 386, 9);
            $vlinha["sequencial"] = substr($linha, 395, 6); //9  Sequencial do registro
        } elseif ($this->tamanho_linha == 240) {
            $vlinha["banco"] = substr($linha, 1, 3); //numerico  //Codigo do Banco na Compensacao
            $vlinha["lote"] = substr($linha, 4, 4); // num - default 9999  //Lote de Servico
            $vlinha["registro"] = substr($linha, 8, 1); //num - default 9   //Tipo de Registro
            $vlinha["cnab1"] = substr($linha, 9, 9); //alpha - default brancos //Uso Exclusivo FEBRABAN/CNAB
            $vlinha["quant_lotes"] = substr($linha, 18, 6); //num. //Quantidade de Lotes do Arquivo
            $vlinha["quant_regs"] = substr($linha, 24, 6); //num. //Quantidade de Registros do Arquivo
            $vlinha["quant_contas_conc"] = substr($linha, 30, 6); //num. //Qtde de Contas p/ Conc. (Lotes)
            $vlinha["cnab2"] = substr($linha, 36, 205); //alpha - default brancos  //Uso Exclusivo FEBRABAN/CNAB
        }
        return $vlinha;
    }

    protected function processarLinhaTrailerLote($linha)
    {
        $vlinha = array();
        $vlinha["banco"] = substr($linha, 1, 3); //numerico  //Codigo do Banco na Compensacao
        $vlinha["lote"] = substr($linha, 4, 4); //numerico //Lote de Servico
        $vlinha["registro"] = substr($linha, 8, 1); //num - default 5 //Tipo de Registro
        $vlinha["cnab1"] = substr($linha, 9, 9); //alfa - default brancos Uso Exclusivo FEBRABAN/CNAB
        $vlinha["quant_regs"] = substr($linha, 18, 6); //numerico //Quantidade de Registros do Lote
        $vlinha["valor"] = substr($linha, 24, 16); //numerico, 2 casas decimais  //Somatoria dos Valores
        $vlinha["quant_moedas"] = substr($linha, 42, 13); //numerico, 5 casas decimais  //Somatoria de Quantidade de Moedas
        $vlinha["num_aviso_debito"] = substr($linha, 60, 6); //numerico //Numero Aviso de Debito
        $vlinha["cnab2"] = substr($linha, 66, 165); //alfa, default brancos //Uso Exclusivo FEBRABAN/CNAB
        $vlinha["ocorrencias"] = substr($linha, 231, 10); //alfa  //Codigos das Ocorrencias para Retorno
        return $vlinha;
    }

    /**
     * Formata uma string, contendo um valor real (float) sem o separador de decimais,
     * para a sua correta representacao real.
     *
     * @param string $valor String contendo o valor na representacao usada nos arquivos
     * de retorno do banco, sem o separador de decimais.
     * @param int $numCasasDecimais Total de casas decimais do n?mero representado
     * em $valor.
     * @return float Retorna o numero representado em $valor, no seu formato float,
     * contendo o separador de decimais.
     * @author Manoel Campos da Silva Filho.
     */
    protected function formataNumero($valor, $numCasasDecimais = 2)
    {
        if ($valor == "")
            return 0;

        $casas = $numCasasDecimais;
        if ($casas > 0) {
            $valor = substr($valor, 0, strlen($valor) - $casas) . "." . substr($valor, strlen($valor) - $casas, $casas);
            $valor = (float)$valor;
        } else
            $valor = (int)$valor;

        return $valor;
    }

    /**
     * Formata uma string, contendo uma data sem o separador, no formato DDMMAA,
     * para o formato DD/MM/AAAA.
     *
     * @param string $data String contendo a data no formato DDMMAA.
     * @return string Retorna a data non formato DD/MM/AAAA.
     * @author Manoel Campos da Silva Filho.
     */
    protected function formataData($data)
    {
        if ($data == "" || $data == '000000')
            return "";

        //formata a data par ao padrao americano MM/DD/AA
        $data = substr($data, 2, 2) . "/" . substr($data, 0, 2) . "/" . substr($data, 4, 2);

        //formata a data, a partir do padrao americano, para o padrao DD/MM/AAAA
        return date("d/m/Y", strtotime($data));
    }

    /**
     * Gera a parcela no acade finan com os dados do retorno como uma conta a receber recebida
     * @param $retorno
     * @return bool
     */
    public static function getIntegracao($retorno)
    {
        $dados = Parcela::getDefaultAdapter()->select()
            ->from(array('p' => TB_PARCELA), null)
            ->join(array('c' => TB_CONTRATO), 'p.id_contrato = c.id', null)
            ->join(array('l' => TB_LOTES), 'c.id_lote = l.id', null)
            ->join(array('e' => TB_EMPREENDIMENTO), 'e.id = l.id_empreendimento', '*')
            ->where('e.id = ' . $retorno['id_empreendimento'])
            ->query()->fetch();

        if ($dados['fl_integracao'] == '1') {
            return true;
        } else {
            return false;
        }
    }

    public static function setUpdateParcelaApi($dt_pagamento, $pathPdf, $pathPdfName)
    {

        //Convertendo Data de Pagamento em padro do MYSQL
        $dt_pagamento = Helper::dataParaAmericano($dt_pagamento);
        //Consulta na tabela REMESSA_ITENS qual o nmero da parcela se refere.
        $tabelaParcela = new Parcela();
        //$listaParcela = $tabelaParcela->getParcelasByIdRemessa($idremessa);
        $listaParcela = $tabelaParcela->getParcelasByDt_pagamento($dt_pagamento);

        //Realiza um loop com os dados das parcelas selecionadas
        // e consulta a api para obter os dados dos boletos (se foi pago ou no) e salva no banco de dados.
        $numerosremessa = [];
        $array_retorno = [];
        $array_ok = [];
        $array_erros = [];
        $array_todos = [];
        foreach ($listaParcela as $lista) {
            //Obtem os dados da parcela
            //Pega os dados do Layout da Api
            $tabelaLayout = new Layout();
            $listaLayout = $tabelaLayout->getTabelaById($lista['cod_layout_api']);

            //Seta o endereo da API
            $uri = $listaLayout['url_api'];
            $uri_param = $listaLayout['uri_param'];
            $urilistar = $listaLayout['url_api_listar'];
            //Seta o tipo de Header que ser enviado.
            $tipo_header = $listaLayout['tipo_header'];
            $tipo_layout = $listaLayout['tipo_layout'];
            $gerar_token = $listaLayout['gerar_token'];
            $pp_token = $listaLayout['pp_token'];
            $password = $listaLayout['password'];

            $dados_layout_itens = LayoutItens::getLayouItenstById($lista['cod_layout_api'], '1');
            $dados_body = explode(";", $dados_layout_itens[0]['conteudo']); //Pega a linha contedo e quebra a expresso regular

            //Realiza a consulta na api e retorna os boletos com o status se foi pago ou no
            //Varivel que armazena o cdigo nico do boleto na API
            $cod_boleto = $lista['id_parcela_contrato'];
            $cod_barras = $lista['cod_barras'];
            $boleto_id = $lista['cod_boleto_api'];

            if ($tipo_layout === 'APIASAAS') {
                $urilistar = $uri;
            }

            $boleto = API::listaBoletosPagos($urilistar, $uri_param, $gerar_token, $pp_token, $password, $cod_boleto, $cod_barras, $dados_body, $boleto_id, $tipo_layout);

            $boletos = json_decode($boleto, true);


            if ($tipo_layout === 'APIASAAS') {
                $vencimento = $boletos['dueDate'];
                $data_pag = $boletos['paymentDate'];
                $data_pagamento_boleto = $boletos['paymentDate'];
                $codigo_status = $boletos['status'];
                $boleto_data_criacao = $boletos['dateCreated'];
                $boleto_multa = $boletos['interestValue'];

            } else {
                foreach ($boletos as $key => $value) {
                    $lista[$key] = $value;

                }
                $data = array_chunk($lista['data'], 1, false);

                //Lista o ultimo Boleto Gerado para o id da parcela.
                // Descarta Boletos j gerados anteriormente
                $id_boleto = $data[0][0]['boleto_id'];
                $cod_barras = $data[0][0]['boleto_codigo_barras'];

                //Remove os pontos e espaos da linha digitvel do boleto
                $linha_digitavel = $data[0][0]['boleto_linha_digitavel'];
                $linha_digitavel = str_replace('.', '', $linha_digitavel);
                $linha_digitavel = str_replace(' ', '', $linha_digitavel);

                $vencimento = $data[0][0]['boleto_vencimento'];
                $data_pag = $data[0][0]['boleto_data_pagamento'];
                $codigo_status = $data[0][0]['boleto_status_id'];
                $valor_boleto_pago = $data[0][0]['boleto_valor_pago'];
                $data_pagamento_boleto = $data[0][0]['boleto_data_pagamento'];
                $boleto_desconto = $data[0][0]['boleto_desconto'];
                $boleto_juros = $data[0][0]['boleto_juros'];
                $boleto_multa = $data[0][0]['boleto_multa'];
                $boleto_status_descricao = $data[0][0]['boleto_status_descricao'];
                $boleto_data_criacao = $data[0][0]['boleto_created_at'];


                //Atualiza a tabela de Parcelas com os dados do boleto.
                //Caso o boleto tenha sido pago atualiza com o valor pago.

                for ($i = 0; $i < count($dados_body); $i++) {
                    $expressao2 = explode(":", $dados_body[$i]); //Quebra a expresso pegando a primeira palavra
                    $posicaoi = explode("-", $expressao2[1]); //Quebra a expresso novamente pegando a coluna inicial
                    $posicaof = explode("+", $posicaoi[1]); //Quebra a expresso novamente pegando a coluna final e o preenchimento
                    $valor = explode(".", $posicaof[1]); //Quebra a expresso novamente pegando o valor definido caso exista

                    //Caso o cdigo gravado na tabela LAYOUT_ITENS seja o mesmo cdigo retornado pela varivel $codigo_status, salva o boleto como pago
                    if ((strpos($expressao2[0], 'cod_boleto_pago') !== false) && ($valor[1] == $codigo_status)) {
                        $auxTabela = self::getDefaultAdapter();
                        $auxTabela->update(array('tp' => TB_PARCELA), array('pago' => '1',
                            // 'dt_geracao_boleto' => '"'.$boleto_data_criacao.'"',
                            'dt_pagamento' => $data_pagamento_boleto,
                            'vl_multa' => $boleto_multa,
                            'vl_juros' => $boleto_juros,
                            'dt_credito' => $data_pagamento_boleto,
                            // 'vl_total_pago' => ($valor_boleto_pago+$boleto_juros+$boleto_multa)
                        ), 'cod_boleto_api = "' . $boleto_id . '"');

                        //Adiciona no Array o nmero do id_parcela_contrato para gerar o relatrio de Retorno de Boleto com as parcelas com pagamento com sucesso
                        array_push($array_ok, $cod_boleto);
                    }
                    if ((strpos($expressao2[0], 'cod_boleto_retido') !== false) && ($valor[1] == $codigo_status)) {
                        $auxTabela = self::getDefaultAdapter();
                        $auxTabela->update(array('tp' => TB_PARCELA), array('pago' => '1',
                            //  'dt_geracao_boleto' => $boleto_data_criacao,
                            'dt_pagamento' => $data_pagamento_boleto,
                            'vl_multa' => $boleto_multa,
                            'vl_juros' => $boleto_juros,
                            'dt_credito' => $data_pagamento_boleto,
                            //'vl_total_pago' => ($valor_boleto_pago+$boleto_juros+$boleto_multa),
                        ), 'cod_boleto_api =  "' . $boleto_id . '"');

                        //Adiciona no Array o nmero do id_parcela_contrato para gerar o relatrio de Retorno de Boleto com as parcelas com pagamento com sucesso
                        array_push($array_ok, $cod_boleto);
                    }
                    if ((strpos($expressao2[0], 'cod_boleto_liquidado') !== false) && ($valor[1] == $codigo_status)) {
                        $auxTabela = self::getDefaultAdapter();
                        $auxTabela->update(array('tp' => TB_PARCELA), array('pago' => '1',
                            // 'dt_geracao_boleto' => $boleto_data_criacao,
                            'dt_pagamento' => $data_pagamento_boleto,
                            'vl_multa' => $boleto_multa,
                            'vl_juros' => $boleto_juros,
                            'dt_credito' => $data_pagamento_boleto,
                            // 'vl_total_pago' => ($valor_boleto_pago+$boleto_juros+$boleto_multa),
                        ), 'cod_boleto_api =  "' . $boleto_id . '"');
                        array_push($array_ok, $cod_boleto);
                    }
                    if ((strpos($expressao2[0], 'cod_boleto_marcado') !== false) && ($valor[1] == $codigo_status)) {
                        $auxTabela = self::getDefaultAdapter();
                        $auxTabela->update(array('tp' => TB_PARCELA), array('pago' => '1',
                            //  'dt_geracao_boleto' => $boleto_data_criacao,
                            'dt_pagamento' => $data_pagamento_boleto,
                            'vl_multa' => $boleto_multa,
                            'vl_juros' => $boleto_juros,
                            'dt_credito' => $data_pagamento_boleto,
                            // 'vl_total_pago' => ($valor_boleto_pago+$boleto_juros+$boleto_multa),
                        ), 'cod_boleto_api =  "' . $boleto_id . '"');
                        array_push($array_ok, $cod_boleto);
                    }
                    array_push($array_todos, $cod_boleto);


                }

                //Cria um array com otodos os nmeros de remessas das parcelas geradas
                array_push($numerosremessa, $lista['id_remessa']);
            }

            //Remove nmero de remessas duplicadas do Array
            $numerosremessa2 = array_unique($numerosremessa, SORT_NUMERIC);

            //Atualizando a remessa com os valores do array $numerosremessa2
            foreach ($numerosremessa2 as $item) {

                //Obtendo dados bancrios de acordo com os dados contidos na remessa
                $tabelaRemessa = new RemessaBoleto();
                $listaRemessa = $tabelaRemessa->getRemessaByBanco($item);

                //Atualiza o status da Remessa e retira ela do Select de remessas em aberto do menu Retorno Boleto
                $auxTabela = $tabelaRemessa->getAdapter();
                $auxTabela->update(array('tp' => TB_REMESSA), array('tp_status' => 'EB'), 'id = ' . $item);

                $cd_banco = $listaRemessa[0]['id_banco'];
                $cd_banco_conta = $listaRemessa[0]['id_banco_conta'];
                $idemp = $listaRemessa[0]['id_empreendimento'];
                $dataGeracao = $listaRemessa[0]['dtGeracao'];
                $tabelabanco = new Banco();
                $listaBanco = $tabelabanco->findByBancoConta($cd_banco_conta);

                //Insero na tabela Retorno Boleto
                $dadosArquivos = new RetornoBoletoTabela();
                $dadosArquivos = $dadosArquivos->createRow();
                $dadosArquivos->cd_banco = str_pad($cd_banco, 3, '0', STR_PAD_LEFT); //Completa com Zeros caso o nmero do Banco no atenda a 3 algarismos
                $dadosArquivos->nr_retorno = $item;
                $dadosArquivos->dt_criacao = $dataGeracao;
                $dadosArquivos->dt_processamento = date('Y-m-d');
                $dadosArquivos->ds_conta = 'Ag: ' . substr($listaBanco[0]['agencia'], -6) . '-' . $listaBanco[0]['agencia_dv'] . ' Cc: ' . substr($listaBanco[0]['conta_corrente'], -8) . '-' . $listaBanco[0]['conta_corrente_dv'];
                $dadosArquivos->nm_cliente = $listaBanco[0]['nm_cedente'];
                $dadosArquivos->ds_arquivo_pdf = $pathPdf . '/' . $pathPdfName;
                $dadosArquivos->ds_arquivo = '';
                $dadosArquivos->save();

            }
            $array_retorno['ok'] = $array_ok;

            $array_todos = array_unique($array_todos, SORT_NUMERIC);
            $array_erros = array_diff($array_todos, $array_ok);
            $array_retorno['erros'] = $array_erros;
            return $array_retorno;
        }
    }


    public static function atualizarParcelaApiIdParcelaContrato($id_parcela_contrato,$valor_nominal,$dt_pagamento, $multa=0, $juros=0,$desconto=0,$nossoNumero = 0, $totalRecebimento = false, $banco = null)
    {        
        $db = self::getDefaultAdapter();
        $db->getProfiler()->setEnabled(true);
        $auxTabela = self::getDefaultAdapter();
        try{
            $auxTabela->update(array('tp' => TB_PARCELA), array(
                'pago' => '1',
                'tp_baixa' => 'B',
                'fm_baixa' => 'BL',
                'dt_pagamento' => $dt_pagamento,
                'vl_multa' => $multa,
                'vl_juros' => $juros,
                'desconto' => $desconto,
                'dt_credito' => $dt_pagamento,
                'nosso_numero' => $nossoNumero,
                'id_banco' => $banco,
                'vl_total_pago' => $totalRecebimento?$totalRecebimento:($valor_nominal+$juros+$multa-$desconto),
                'obs_baixa' => 'Baixa feita atravs da API '.date('d/m/Y H:i:s')
            ), 'id_parcela_contrato = '. $id_parcela_contrato);
        }
        catch(Exception $e){
            var_dump($db->getProfiler()->getLastQueryProfile()->getQuery());
            var_dump($db->getProfiler()->getLastQueryProfile()->getQueryParams());
            $db->getProfiler()->setEnabled(false);    
            die;
        }   
    }

    public static function atualizarParcelaApiIdParcelaContratoBancoGrafeno($id_parcela_contrato,$valor_nominal,$dt_pagamento, $multa=0, $juros=0,$desconto=0,$nossoNumero = 0, $totalRecebimento = false, $banco = null)
    {        
        $db = self::getDefaultAdapter();
        $db->getProfiler()->setEnabled(true);
        $auxTabela = self::getDefaultAdapter();
        try{
            $auxTabela->update(array('tp' => TB_PARCELA), array(
                'pago' => '1',
                'tp_baixa' => 'B',
                'fm_baixa' => 'BL',
                'dt_pagamento' => $dt_pagamento,
                'vl_multa' => $multa,
                'vl_juros' => $juros,
                'desconto' => $desconto,
                'dt_credito' => $dt_pagamento,
                'vl_total_pago' => $totalRecebimento,
                'obs_baixa' => $totalRecebimento?'Baixa feita atravs do Webhook do banco Grafeno '.date('d/m/Y H:i:s'):'Baixa feita atravs da API '.date('d/m/Y H:i:s'),
                'id_banco' => $banco,
            ), 'id_parcela_contrato = '. $id_parcela_contrato . ' or id_parcela_contrato_antigo = ' . $id_parcela_contrato);
        }
        catch(Exception $e){
            var_dump($db->getProfiler()->getLastQueryProfile()->getQuery());
            var_dump($db->getProfiler()->getLastQueryProfile()->getQueryParams());
            $db->getProfiler()->setEnabled(false);    
            die;
        }   
    }
}
?>