<?php

/*
 * @descr: Gera o arquivo de remessa para cobranca no padrao CNAB 240 vers. 10.01 FEBRABAN
 */

class RemessaInstrucao extends Zend_Db_Table_Abstract
{
    protected $_name = TB_REMESSA_INSTRUCAO;

    /**
     * Limita e trata as palavras e numeros para o formato dos campos
     * @param $palavra
     * @param $limite
     * @param $comp
     * @param $antecede
     * @return string
     */
    private static function limit($palavra, $limite, $comp = 'brancos', $antecede = false, $caracterEspecial = true)
    {
        $var = '';
        if ($caracterEspecial)
            $palavra = strtoupper(self::retirar_caracteres_especiais($palavra));

        if (strlen($palavra) >= $limite) {
            $var = substr($palavra, 0, $limite);
        } else {
            $max = (int)($limite - strlen($palavra));
            if ($comp == 'brancos') {
                if ($antecede) {
                    $var = self::complementoRegistro($max, "brancos") . $palavra;
                } else {
                    $var = $palavra . self::complementoRegistro($max, "brancos");
                }
            } elseif ($comp == 'zeros') {
                $var = self::complementoRegistro($max, "zeros") . $palavra;
            }
        }
        return $var;
    }

    public static function modulo_10($num)
    {
        $numtotal10 = 0;
        $fator = 2;

        // Separacao dos numeros
        for ($i = strlen($num); $i > 0; $i--) {
            // pega cada numero isoladamente
            $numeros[$i] = substr($num, $i - 1, 1);
            // Efetua multiplicacao do numero pelo (falor 10)
            // 2002-07-07 01:33:34 Macete para adequar ao Mod10 do Itau
            $temp = $numeros[$i] * $fator;
            $temp0 = 0;
            foreach (preg_split('//', $temp, -1, PREG_SPLIT_NO_EMPTY) as $k => $v) {
                $temp0 += $v;
            }
            $parcial10[$i] = $temp0; //$numeros[$i] * $fator;
            // monta sequencia para soma dos digitos no (modulo 10)
            $numtotal10 += $parcial10[$i];
            if ($fator == 2) {
                $fator = 1;
            } else {
                $fator = 2; // intercala fator de multiplicacao (modulo 10)
            }
        }

        // varias linhas removidas, vide funcao original
        // Calculo do modulo 10
        $resto = $numtotal10 % 10;
        $digito = 10 - $resto;
        if ($resto == 0) {
            $digito = 0;
        }

        return $digito;

    }

    public static function modulo_11($num, $base = 9, $r = 0)
    {
        $soma = 0;
        $fator = 2;

        /* Separacao dos numeros */
        for ($i = strlen($num); $i > 0; $i--) {
            // pega cada numero isoladamente
            $numeros[$i] = substr($num, $i - 1, 1);
            // Efetua multiplicacao do numero pelo falor
            $parcial[$i] = $numeros[$i] * $fator;
            // Soma dos digitos
            $soma += $parcial[$i];
            if ($fator == $base) {
                // restaura fator de multiplicacao para 2
                $fator = 1;
            }
            $fator++;
        }

        /* Calculo do modulo 11 */
        if ($r == 0) {
            $soma *= 10;
            $digito = $soma % 11;
            if ($digito == 10) {
                $digito = 0;
            }
            return $digito;
        } elseif ($r == 1) {
            $resto = $soma % 11;
            return $resto;
        }
        return false;
    }

    /**
     * Transforma o int do sequencial para o padrao febraban
     * @param $i
     * @param $length
     * @return string
     */
    private static function sequencial($i, $length)
    {
        return str_pad($i, $length, 0, STR_PAD_LEFT);
    }

    /**
     * Complementa os registros de acordo com as normas do febraban
     * @param $int
     * @param $tipo ('zeros','brancos')
     * @return string
     */
    private static function complementoRegistro($int, $tipo)
    {
        $space = '';
        if ($tipo == "zeros") {
            for ($i = 1; $i <= $int; $i++) {
                $space .= '0';
            }
        } else if ($tipo == "brancos") {
            for ($i = 1; $i <= $int; $i++) {
                $space .= ' ';
            }
        }
        return $space;
    }

    /**
     * Remove os caracteres especiais de uma string
     * @param $string
     * @return mixed
     */
    private static function retirar_caracteres_especiais($string)
    {
        $palavra = Helper::removeAcentos($string);
        $palavranova = str_replace("_", " ", $palavra);
        $palavranova = str_replace(array("-", ",", "."), "", $palavranova);
        return $palavranova;
    }

    /**
     * Funcao para calcular o digito desconsiderando o '10' e inserindo 'P' no lugar
     * @param $numero
     * @return int|string
     */
    public static function digitoVerificadorP($numero)
    {
        $resto2 = self::modulo_11($numero, 7, 1);
        $digito = 11 - $resto2;
        if ($digito == 10) {
            $dv = "P";
        } elseif ($digito == 11) {
            $dv = 0;
        } else {
            $dv = $digito;
        }
        return $dv;
    }

    public static function digitoVerificadorRibeiraoPreto($agencia, $carteira, $nossoNumero)
    {
        $numeros = str_split($agencia.$carteira.$nossoNumero);
        $multiplicador = 2;
        $novosNumeros = array();

        foreach ($numeros as $numero) {

            $novosNumeros[] = $multiplicador * $numero;
            
            if ($multiplicador == 2)
                $multiplicador = 1;
            else 
                $multiplicador = 2;
        }

        $somaNumeros = 0;

        foreach ($novosNumeros as $novoNumero) {

            if ($novoNumero > 9) {
                $divideNumero = str_split($novoNumero);
                
                $somaNumeroDividido = 0;
                
                foreach ($divideNumero as $divide) {
                    $somaNumeroDividido += (int)$divide;
                }
                
                $somaNumeros += $somaNumeroDividido;
                continue;
            }

            $somaNumeros += $novoNumero;
        }
        $resto = $somaNumeros % 10;

        if ($resto == 0)
            $digitoVerificador = 0;
        else
            $digitoVerificador = 10 - $resto;

        return $digitoVerificador;
    }

    /**
     * Gera o nosso numero de acordo com o banco
     * @param $banco
     * @param $parcela
     * @param null $adicionais
     * @return string
     */
    private static function geraNossoNumero($banco, $parcela, $emp = null)
    {
        $nossoNumero = '';
        
        switch ($banco) {
            case Banco::BANCO_SANTANDER:
                $nossoNumero = str_pad($parcela['id_parcela_contrato'], 20, '0', STR_PAD_LEFT);

                if ($parcela['nr_boleto']) {
                    $nossoNumero = str_pad($parcela['nr_boleto'], 20, '0', STR_PAD_LEFT);
                }

                break;
            case Banco::BANCO_BRADESCO:
                $numeroCalculo = str_pad($emp['carteira'], 2, '0', STR_PAD_LEFT) .
                    str_pad($parcela['id_parcela_contrato'], 11, '0', STR_PAD_LEFT);
                $nossoNumero = str_pad($emp['carteira'], 3, '0', STR_PAD_LEFT) .
                    self::complementoRegistro(5, 'zeros') .
                    str_pad($parcela['id_parcela_contrato'], 11, '0', STR_PAD_LEFT);
                $nossoNumero = $nossoNumero . self::digitoVerificadorP($numeroCalculo);

                if ($parcela['nr_boleto']) {
                    $numeroCalculo = str_pad($emp['carteira'], 2, '0', STR_PAD_LEFT) .
                    str_pad($parcela['nr_boleto'], 11, '0', STR_PAD_LEFT);
                    $nossoNumero = str_pad($emp['carteira'], 3, '0', STR_PAD_LEFT) .
                        self::complementoRegistro(5, 'zeros') .
                        str_pad($parcela['nr_boleto'], 11, '0', STR_PAD_LEFT);
                    $nossoNumero = $nossoNumero . self::digitoVerificadorP($numeroCalculo);
                }
                break;
            case Banco::BANCO_RIBEIRAO_PRETO:
                $numeroCalculo = str_pad($emp['carteira'], 2, '0', STR_PAD_LEFT) .
                    str_pad($parcela['nr_boleto'], 11, '0', STR_PAD_LEFT);
                $nossoNumero = str_pad($emp['carteira'], 3, '0', STR_PAD_LEFT) .
                    self::complementoRegistro(5, 'zeros') .
                    str_pad($parcela['nr_boleto'], 11, '0', STR_PAD_LEFT);
                $nossoNumero = $nossoNumero . self::digitoVerificadorP($numeroCalculo);
                $nossoNumero[8] = $nossoNumero[2];
                $nossoNumero[7] = $nossoNumero[1];
                $nossoNumero[6] = $nossoNumero[0];
                $nossoNumero[1] = 0;
                break;
            case Banco::BANCO_DO_BRASIL:
                if (strlen($emp['cd_cedente']) == 4) {
                    $nossoNumero = $emp['cd_cedente'] . str_pad($parcela['id_parcela_contrato'], 7, '0', STR_PAD_LEFT);

                    if ($parcela['nr_boleto']) {
                        $nossoNumero = $emp['cd_cedente'] . str_pad($parcela['nr_boleto'], 7, '0', STR_PAD_LEFT);
                    }

                    $nossoNumero = $nossoNumero . self::modulo_11($nossoNumero);
                } elseif (strlen($emp['cd_cedente']) == 6) {
                    $nossoNumero = $emp['cd_cedente'] . str_pad($parcela['id_parcela_contrato'], 5, '0', STR_PAD_LEFT);

                    if ($parcela['nr_boleto']) {
                        $nossoNumero = $emp['cd_cedente'] . str_pad($parcela['nr_boleto'], 5, '0', STR_PAD_LEFT);
                    }

                    $nossoNumero = $nossoNumero . self::modulo_11($nossoNumero);
                } elseif (strlen($emp['cd_cedente']) == 7) {
                    $nossoNumero = $emp['cd_cedente'] . str_pad($parcela['id_parcela_contrato'], 10, '0', STR_PAD_LEFT);

                    if ($parcela['nr_boleto']) {
                        $nossoNumero = $emp['cd_cedente'] . str_pad($parcela['nr_boleto'], 10, '0', STR_PAD_LEFT);
                    }
                } else {
                    $nossoNumero = self::complementoRegistro(15, 'zeros');
                }
                $nossoNumero = self::limit($nossoNumero, 20, 'brancos');
                break;
            case Banco::BANCO_SICREDI:
                $nossoNumero = date('y').substr($parcela['nr_boleto'], -6);
                
                if ($parcela['ano_boleto_emissao'])
                    $nossoNumero = $parcela['ano_boleto_emissao'].substr($parcela['nr_boleto'], -6);

                $dvNossoNumero = self::modulo_11($emp['agencia'].$emp['carteira_variacao'].$emp['cd_cedente'].$nossoNumero);
                $nossoNumero = self::limit($nossoNumero.$dvNossoNumero, 20, 'brancos');
                break;
            case Banco::BANCO_ITAU:
                $nossoNumero = str_pad((int)$parcela['id_parcela_contrato'], 8, '0', STR_PAD_LEFT);

                if ($parcela['nr_boleto']) {
                    $nossoNumero = str_pad((int)$parcela['nr_boleto'], 8, '0', STR_PAD_LEFT);
                }

                $nossoNumero = $emp['carteira'] . $nossoNumero . self::modulo_10($nossoNumero);
                $nossoNumero = $nossoNumero . self::complementoRegistro(8, 'brancos');
                break;
            case Banco::BANCO_CAIXA_ECONOMICA:
                $nossoNumero = '00014' . str_pad($parcela['id_parcela_contrato'], 15, '0', STR_PAD_LEFT);
                if($parcela['nr_boleto']) {
                    $nossoNumero = '00014' . str_pad($parcela['nr_boleto'], 15, '0', STR_PAD_LEFT);
                }
                break;
            case Banco::BANCO_SICOOB:
                $nossoNumero = str_pad((int)$parcela['id_parcela_contrato'], 9, '0', STR_PAD_LEFT);
                if ($parcela['nr_boleto']) {
                    $nossoNumero = str_pad((int)$parcela['nr_boleto'], 9, '0', STR_PAD_LEFT);
                }
                $nossoNumero .= self::digitoVerificadorSicoob($emp['agencia'],$emp['cd_cedente'],$nossoNumero);
                $nossoNumero .= '01014';
                $nossoNumero .= self::complementoRegistro(5, 'brancos');
                break;
            case Banco::BANCO_NORDESTE:
                $nossoNumero = str_pad((int)$parcela['id_parcela_contrato'], 7, '0', STR_PAD_LEFT);
                if ($parcela['nr_boleto']) {
                    $nossoNumero = str_pad((int)$parcela['nr_boleto'], 7, '0', STR_PAD_LEFT);
                }
                $nossoNumero .= self::modulo_11($nossoNumero);
                $nossoNumero .= self::complementoRegistro(12, 'brancos');
                break;
            case Banco::BANCO_BANPARA:
                $nossoNumero = '00000' . str_pad($parcela['id_parcela_contrato'], 15, '0', STR_PAD_LEFT);
                if ($parcela['nr_boleto']) {
                    $nossoNumero = '00000' . str_pad($parcela['nr_boleto'], 15, '0', STR_PAD_LEFT);
                }
                break;
            case Banco::BANCO_SAFRA:
                $nossoNumero = str_pad($parcela['id_parcela_contrato'], 10, '0', STR_PAD_LEFT);
                if ($parcela['nr_boleto']) {
                    $nossoNumero = str_pad($parcela['nr_boleto'], 10, '0', STR_PAD_LEFT);
                }
                break;
            case Banco::BANCO_UNIPRIME:
                $numeroCalculo = str_pad($emp['carteira'], 2, '0', STR_PAD_LEFT).str_pad($parcela['nr_boleto'], 11, '0', STR_PAD_LEFT);
                $nossoNumeroN = ltrim($parcela['nr_boleto'].self::digitoVerificadorP($numeroCalculo),'0');
                $nossoNumero = str_pad(self::limit($nossoNumeroN, 12, 'zeros'), 20, ' ', STR_PAD_RIGHT);
                break;
        }
        return $nossoNumero;
    }

    /**
     * calcula o digito verificador do Sicoob
     */
    private static function digitoVerificadorSicoob($agencia,$cedente,$nosso_numero) {
        
        $cedente = self::complementoRegistro(10 - strlen(str_replace("-","",$cedente)), "zeros") . '' . $cedente;
        $nosso_numero = substr($nosso_numero, 2, 7);
        $Dv = '';

        if ($nosso_numero && $agencia && $cedente)
        {
            $NossoNumero = self::formataNumDoc((int)$nosso_numero,7);
            $sequencia = self::formataNumDoc($agencia,4).self::formataNumDoc(str_replace("-","",$cedente),10).self::formataNumDoc($NossoNumero,7);
            
            $cont=0;
            $calculoDv = '';

            for($num = 0; $num <= strlen($sequencia); $num++)
            {
                $cont++;
                // constante fixa Sicoob e 3197
                if($cont == 1)
                    $constante = 3;
                elseif($cont == 2)
                    $constante = 1;
                elseif($cont == 3)
                    $constante = 9;
                elseif($cont == 4){
                    $constante = 7;
                    $cont = 0;
                }
                
                $calculoDv = (float)$calculoDv + (float)(substr($sequencia, $num, 1) * (int)$constante);
            }

            $Resto = $calculoDv % 11;
            $Dv = 11 - $Resto;
            $Dv = ($Dv > 9) ? 0 : $Dv;
        }
        
        return $Dv;
    }

    private static function formataNumDoc($num,$tamanho)
    {
        while(strlen($num)<$tamanho)
        {
            $num="0".$num;
        }
        return $num;
    }

    //Altera o versionamento do arquivo de acordo com o banco utilizado
    private static function versionamentoLayout($banco, $cdCedente = null)
    {
        $versaoH = '060';
        $versaoL = '030';
        switch ($banco) {
            case Banco::BANCO_SANTANDER:
                $versaoH = '040';
                break;
            case Banco::BANCO_BRADESCO:
                $versaoH = '084';
                $versaoL = '042';
                break;
            case Banco::BANCO_RIBEIRAO_PRETO:
                $versaoH = '084';
                $versaoL = '030';
                break;
            case Banco::BANCO_DO_BRASIL:
                $versaoH = '083';
                $versaoL = '042';
                break;
            case Banco::BANCO_SICREDI:
                $versaoH = '081';
                $versaoL = '040';
                break;
            case Banco::BANCO_ITAU:
                $versaoH = '040';
                break;
            case Banco::BANCO_CAIXA_ECONOMICA:
                
                if (strlen($cdCedente) < 7) {
                    $versaoH = '101';
                    $versaoL = '060';
                } else {
                    $versaoH = '107';
                    $versaoL = '067';
                }

                break;
            case Banco::BANCO_SICOOB:
                $versaoL = '040';
                $versaoH = '081';
                break;
            case Banco::BANCO_BANPARA:
                $versaoH = '101';
                $versaoL = '060';
                break;
            case Banco::BANCO_SAFRA:
                $versaoH = '103';
                $versaoL = '060';
                break;
        }
        return array('header' => $versaoH, 'lote' => $versaoL);
    }

    /**
     * Verifica os codigos de carteira para os bancos especificamente
     * @param $banco
     * @param $carteira
     * @return string
     */
    private static function verificaCodigoCarteira($banco, $carteira)
    {
        $codCarteira = '1';
        switch ($banco) {
            case Banco::BANCO_SANTANDER:
                $codCarteira = '5';
                break;
            case Banco::BANCO_BRADESCO:
                $codCarteira = '1';
                break;
            case Banco::BANCO_RIBEIRAO_PRETO:
                $codCarteira = '1';
                break;
            case Banco::BANCO_DO_BRASIL:
                if ($carteira == '11' || $carteira == '12') {
                    $codCarteira = '1';
                } elseif ($carteira == '17') {
                    $codCarteira = '7';
                } else {
                    $codCarteira = '1';
                }
                break;
            case Banco::BANCO_SICREDI:
                //Numero da unicred possui 2 digitos entao o primeiro digito precisa estar anexado ao nosso numero
                $codCarteira = '1';
                break;
            case Banco::BANCO_ITAU:
                $codCarteira = '1';
                break;
            case Banco::BANCO_CAIXA_ECONOMICA:
                $codCarteira = '1';
                break;
        }
        return $codCarteira;
    }

    /**
     * Gera O registro no banco da remessa de alteracao de registros ja gerados
     * @param $idContrato
     * @param string $codMovimento
     * @return mixed
     */
    public static function geraRemessaInstrucao($idContrato, $codMovimento = '01')
    {
        $parcelas = Parcela::getInstance()->getAdapter()->select()
            ->from(array('p' => TB_PARCELA), array('id_parcela_contrato', 'id_contrato', 'vl_parcela', 'dt_parcela', 'id_remessa', 'desconto', 'acrescimo', 'nr_boleto'))
            ->join(array('c' => TB_CONTRATO), 'p.id_contrato = c.id', null)
            ->join(array('l' => TB_LOTES), 'c.id_lote = l.id', null)
            ->join(array('e' => TB_EMPREENDIMENTO), 'e.id = l.id_empreendimento', array('idEmpreendimento' => 'id'))
            ->join(array('pe' => TB_PESSOA), 'pe.id = c.id_pessoa', array('nm_pessoa', 'endereco', 'nr_cep', 'nm_bairro', 'tp_pessoa'))
            ->joinLeft(array('pf' => TB_PESSOA_FISICA), 'pe.id = pf.id_pessoa', array('nr_cpf'))
            ->joinLeft(array('pj' => TB_PESSOA_JURIDICA), 'pe.id = pj.id_pessoa', array('nr_cnpj'))
            ->joinLeft(array('ci' => TB_CIDADE), 'pe.cd_cidade = ci.id', array('cidade' => 'nome', 'uf'))
            ->where('c.id = ' . $idContrato)
            ->where('p.pago = "0"')
            ->where('p.id_remessa IS NOT NULL')
            ->query()->fetchAll();

        if (count($parcelas) == 0)
            return false;

        //Busca parcelas que ja foram criadas e nao foram enviadas
        $remessaInstrucao = new RemessaInstrucao();
        $idsParcela = array();
        foreach ($parcelas as $k => $parcela) {
            $testeRemessa = $remessaInstrucao->fetchRow('id_remessa_instrucao = ' . $parcela['id_remessa'] . ' AND id_remessa IS NULL');
            if ($testeRemessa)
                unset($parcelas[$k]);
            else {
                $idsParcela[] = $parcela['id_parcela_contrato'];
            }
        }

        //Busca os dados necessarios para gerar o arquivo de lote
        $emp = Empreendimento::getDefaultAdapter()->select()
            ->from(array('emp' => TB_EMPREENDIMENTO), '*')
            ->join(array('b' => TB_BANCO), 'emp.id = b.id_empreendimento', array('*'))
            ->where('emp.id = ' . $parcelas[0]['idEmpreendimento'])
            ->query()->fetch();

        $emp['contrato'] = $idContrato;
        $emp['parcelas'] = $parcelas;

        //Verifica se existe alguma remessa de instrucao aberta para acrescentar o lote
        $remessa = $remessaInstrucao->fetchRow('id_empreendimento = ' . $emp['id'] .
            ' AND id_banco = ' . $emp['cd_banco'] . ' AND id_remessa IS NULL');

        //Calcula o numero de registro para incrementar ou aplicar na criacao da remessa instrucao
        $nrRegistros = (count($parcelas) * 3) + 2;

        if (!$remessa) {
            $remessa = $remessaInstrucao->createRow();
            $remessa->id_empreendimento = $emp['id'];
            $remessa->nr_registro = $nrRegistros + 2;
            $remessa->id_banco = $emp['cd_banco'];
            $remessa->ds_log = (Login::getUsuario()->getLogin() . ' - ' . date('d/m/Y H:m:s') . ' - i');
            $remessa->save();
            $nrLote = 1;
        } else {
            //Se ja existe remessa busca os lotes da mesma para registrar o numero do lote atual
            $lotes = RemessaLote::getInstance()->fetchAll('id_remessa_instrucoes = ' . $remessa->id_remessa_instrucao);
            $nrLote = count($lotes) + 1;
            $remessa->nr_registro = (int)$remessa->nr_registro + ($nrRegistros + 2);
            $remessa->save();
        }

        $detalhe = $remessaInstrucao->gerarDetalhe($codMovimento, $emp, $parcelas, $nrLote, $idContrato);

        $lote = RemessaLote::getInstance()->createRow();
        $lote->id_remessa_instrucoes = $remessa->id_remessa_instrucao;
        $lote->id_contrato = $idContrato;
        $lote->ds_detalhes_registro = $detalhe;
        $lote->ds_codigo_movimento = $codMovimento;
        $lote->ds_ids_parcelas = implode(',', $idsParcela);
        $lote->ds_log = (Login::getUsuario()->getLogin() . ' - ' . date('d/m/Y H:m:s') . ' - D');

        $idLote = $lote->save();

        return $idLote;
    }

    /**
     * Gera arquivo remessa para inclusao sem gravar registro no banco de dados
     * @param array $dados
     * @param string $nmArquivo
     * @param boolean $loteUnico
     * @return mixed
     */
    public static function geraRemessa($dados, $nmArquivo, $loteUnico = false, $tipo_remessa = null, $emissor = 2)
    {
        $contratos = $dados['contratos'];
        if (count($contratos) == 0)
            return false;

        //Cria os diretorios para onde o arquivo vai ser gravado
        $path = explode('.', $_SERVER['HTTP_HOST']);
        if ($path[0] == 'www') {
            $path = $path[1];
        } else {
            $path = $path[0];
        }
        
        $filename = $nmArquivo;
        $filepath = DIRETORIO_DOWNLOAD . "remessas/" . $path;
        $pathAno = DIRETORIO_DOWNLOAD . "remessas/" . $path . '/' . date('Y');
        
        if (is_dir($filepath)) {
            if (!is_dir($pathAno))
                mkdir($pathAno);
        } else {
            mkdir($filepath);
            mkdir($pathAno);
        }
        
        //Grava o conteudo dos registro no corpo do arquivo a ser gerado
        $conteudo = self::gerarHeader($dados);
        $idsParcela = array();
        
        //BRADESCO OU ITAU
        if (in_array($dados['cdBanco'], ['237', '741', '341', '001'])) {
            $nrRegistros = 1;
            $vlTotal = 0.00;
            $nrLote = 1;

            //Gera o header do lote e incrementa o numero de registros
            $conteudo .= self::geraHeaderLote($dados, $nrRegistros);
            
            $nrRegistros++;
            $nrRegistroLote = 1;
            foreach ($contratos as $idContrato) {
                $parcelas = $idContrato['parcelas'];

                //Busca parcelas que ja foram criadas e nao foram enviadas e gera registro para ela
                foreach ($parcelas as $k => $parcela) {
                    if ($parcela['id_remessa'])
                        unset($parcelas[$k]);
                    else {
                        $idsParcela[] = $parcela['id_parcela_contrato'];

                        $conteudo .= self::gerarDetalheApenas('01', $dados, $parcela, $nrLote, $idContrato['id'], $nrRegistroLote, $emissor);

                        //Atualiza contadores de registros e lotes
                        $nrRegistros += 3;
                        $nrRegistroLote += 3;
                        $vlTotal += (float)$parcela['vl_parcela'];
                    }
                }
            }
            $conteudo .= self::gerarTrailerLote($nrRegistros, $dados, $idsParcela, $nrLote, $vlTotal);
        } else {
            //TODOS OUTROS BANCOS
            $nrLote = 0;
            $nrRegistros = 0;
            foreach ($contratos as $idContrato) {
                $nrLote++;
                $parcelas = $idContrato['parcelas'];

                //Busca parcelas que ja foram criadas e nao foram enviadas
                foreach ($parcelas as $k => $parcela) {
                    if ($parcela['id_remessa'])
                        unset($parcelas[$k]);
                    else {
                        if ($parcela['nr_boleto']) {
                            $idsParcela[] = $parcela['nr_boleto'];
                        } else {
                            $idsParcela[] = $parcela['id_parcela_contrato'];
                        }
                    }
                }
                //Atualiza contadores de registros e lotes
                $nrRegistros += (count($parcelas) * 3) + 2;
                $conteudo .= self::gerarDetalhe('01', $dados, $parcelas, $nrLote, $idContrato['id'], $emissor);
            }
        }

        $conteudo .= self::gerarTrailer($dados['cdBanco'], $nrLote, $nrRegistros);

        if (!$handle = fopen($pathAno . '/' . $filename, 'w+')) {
            $msg = "N�o foi poss�vel abrir o arquivo ($filename)";
            $erro = true;
        } // Escreve $conteudo no nosso arquivo aberto.
        elseif (fwrite($handle, "$conteudo") === FALSE) {
            $msg = "N�o foi poss�vel escrever no arquivo ($filename)";
            $erro = true;
        } else {
            $msg = "Arquivo de remessa gerado com sucesso!";
            $erro = false;
        }
        fclose($handle);

        $retorno[0] = $conteudo;
        $retorno[1] = $erro;
        $retorno[2] = $msg;

        return $retorno;
    }

    /**
     * Gera arquivo remessa para envio utilizando os dados de RemessaInstrucao e RemessaLote
     * @param $idRemessa
     * @param $nmArquivo
     * @return mixed
     */
    public static function gerarArquivoInstrucao($idRemessa, $nmArquivo)
    {
        $remInst = new RemessaInstrucao();
        $remessaInstrucao = $remInst->fetchRow('id_remessa_instrucao = ' . $idRemessa);
        $lotesRemessa = RemessaLote::getInstance()->fetchAll('id_remessa_instrucoes = ' . $remessaInstrucao['id_remessa_instrucao']);

        if (count($lotesRemessa) > 0) {

            //Cria os diretorios para onde o arquivo vai ser gravado
            $path = explode('.', $_SERVER['HTTP_HOST']);
            if ($path[0] == 'www') {
                $path = $path[1];
            } else {
                $path = $path[0];
            }
            $filename = $nmArquivo;
            $filepath = DIRETORIO_DOWNLOAD . "remessas/" . $path;
            $pathAno = DIRETORIO_DOWNLOAD . "remessas/" . $path . '/' . date('Y');

            if (is_dir($filepath)) {
                if (!is_dir($pathAno)) mkdir($pathAno);
            } else {
                mkdir($filepath);
                mkdir($pathAno);
            }

            //Grava o conteudo dos registro no corpo do arquivo a ser gerado
            $conteudo = $remessaInstrucao['ds_header'];
            foreach ($lotesRemessa as $lote) {
                $conteudo .= $lote['ds_detalhes_registro'];
            }
            $conteudo .= $remessaInstrucao['ds_trailer'];

            if (!$handle = fopen($pathAno . '/' . $filename, 'w+')) {
                $msg = "N�o foi poss�vel abrir o arquivo ($filename)";
                $erro = true;
            } // Escreve $conteudo no nosso arquivo aberto.
            elseif (fwrite($handle, "$conteudo") === FALSE) {
                $msg = "N�o foi poss�vel escrever no arquivo ($filename)";
                $erro = true;
            } else {
                $msg = "Arquivo de remessa gerado com sucesso!";
                $erro = false;
            }
            fclose($handle);

            $retorno[0] = $msg;
            $retorno[1] = $erro;

        } else {
            $retorno[0] = 'N�o existem parcelas a serem incluidas no arquivo';
            $retorno[1] = true;
        }

        return $retorno;
    }

    /**
     * Gera o Header do arquivo utilizando os dados da tabela remessa instrucao
     * @param $emp
     * @return string
     */
    public static function gerarHeader($emp)
    {
        $cdBanco = $emp['cdBanco'];
        $cnpjCpfCedente = ($emp["cnpj_cedente"])?$emp["cnpj_cedente"]:$emp["cpf_cedente"];

        //apenas para caixa verifica se e pessoa fisica ou juridica e seta 1 ou 2, para outros bancos, seta 2
        $tpInscricao = '2';
        if ($emp['cd_banco'] == '104' || $emp['cd_banco'] == '756' || $emp['cd_banco'] == Banco::BANCO_BRADESCO || $emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO || $emp['cd_banco'] == Banco::BANCO_DO_BRASIL || $emp['cd_banco'] == Banco::BANCO_SAFRA || $emp['cd_banco'] == Banco::BANCO_ITAU || $emp['cd_banco'] == Banco::BANCO_NORDESTE || $emp['cd_banco'] == Banco::BANCO_SANTANDER || $emp['cd_banco'] == Banco::BANCO_SICREDI ) { //Registro complementar para caixa economica
            $tpInscricao = ($emp["cnpj_cedente"])?'2':'1';
        }

        $codEmpresa = str_replace(array(".", "/", "-"), "", $cnpjCpfCedente);
        $empresa = strtoupper(self::retirar_caracteres_especiais($emp["nm_cedente"]));
        $nmBanco = strtoupper(Helper::tirarAcentos(Banco::findBancoNome($cdBanco)));
        $nrRemessa = $emp['idRemessa'];

        $conteudo = '';

        ##  HEADER DO ARQUIVO - REGISTRO 0
        #   NOME DO CAMPO           #SIGNIFICADO            #POSICAO       #PICTURE
        $conteudo .= $cdBanco;                                                          //01  Codigo do banco    caixa 104                   001 003        9(03)
        $conteudo .= '0000';                                                            //02  Lote de servico                                004 007        9(04)
        $conteudo .= '0';                                                               //03  Registro do header                              008 008        9(01)
        $conteudo .= self::complementoRegistro(9, "brancos");                   //04  Uso exclusivo do banco                          009 017        9(09)
        $conteudo .= $tpInscricao;                                                               //05  Tipo de inscricao da construtora                018 018        9(01)
        $conteudo .= self::limit($codEmpresa, 14, "zeros");                             //06  Numero de inscricao da construtora            019 032        9(14)
        if ($emp['cd_banco'] == '001') {
            $conteudo .= self::limit($emp['cd_cedente'], 9, "zeros");           //07a Codigo do cedente do empreendimento            033 041        9(09)
            $conteudo .= '0014';                                                        //07b Codigo da fixo do banco                         042 045        9(04)
            $conteudo .= self::limit($emp['carteira'], 2, "zeros");             //07c Codigo da carteira                              046 047        9(02)
            $conteudo .= self::limit($emp['carteira_variacao'], 3, "zeros");    //07d Codigo da variacao da carteira                  048 050        9(03)
            $conteudo .= self::complementoRegistro(2, "brancos");               //07e Uso exclusivo da banco                          051 052        X(02)
        } elseif ($emp['cd_banco'] == '237' || $emp['cd_banco'] == Banco::BANCO_SAFRA) { //Bradesco,Safra
            $conteudo .= self::limit($emp['cd_cedente'], 20, "zeros");          //07  Codigo do convenio do cedente                   033 052        9(20)
        } elseif ($emp['cd_banco'] == '004' || $emp['cd_banco'] == '104') {     //Banco Itau e Caixa
            $conteudo .= self::complementoRegistro(20, "zeros");                //07 Uso exclusivo da banco                           033 052        X(20)
        } elseif ($emp['cd_banco'] == Banco::BANCO_BANPARA){
            $conteudo .= self::limit($emp['cd_cedente'], 20, "brancos");          //07  Codigo do convenio do cedente
        } elseif ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO) {
            $conteudo .= self::limit('050' . $emp['agencia'] . $emp['agencia_dv'] . $emp['cd_cedente'], 20, "brancos");
        } else {
            $conteudo .= self::complementoRegistro(20, "brancos");              //07  Uso exclusivo da banco                          033 052        9(20)
        }

        $bancoRibeiraoPreto = '';

        if ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO) {
            $bancoRibeiraoPreto .= $emp["agencia"];
        } else {
            $conteudo .= self::limit($emp["agencia"], 5, "zeros");                  //08  Agencia                                         053 057        9(05)
        }

        if ($emp['cd_banco'] == '341') { //Banco Itau
            $conteudo .= self::complementoRegistro(1, "brancos");               //09 Complemento de registro                          058 058        X(01)
        } elseif ($emp['cd_banco'] == '748') { //Banco SICREDI
            $conteudo .= ($emp["agencia_dv"])?$emp["agencia_dv"]:' ';           //09  Digito verificador da agencia (vazio)           058 058        9(01)
        } elseif ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO) {
            $bancoRibeiraoPreto .= $emp["agencia_dv"];
        } else {
            $conteudo .= self::limit($emp["agencia_dv"], 1, "zeros");           //09  Digito verificador da agencia ('0')             058 058        9(01)
        }

        //Codicional se e caixa economica incluindo registro especifico
        if ($cdBanco == '104') {
            
            if (strlen($emp['cd_cedente']) < 7) {
                $conteudo .= self::limit($emp['cd_cedente'], 6, "zeros");           //10  Codigo do convenio                             059 064        9(06)
                $conteudo .= self::complementoRegistro(8, "zeros");                 //11  Uso exclusivo da banco                          065 072        9(08)
            } else {
                $conteudo .= self::limit($emp['cd_cedente'], 7, "zeros");           //10  Codigo do convenio                             059 065        9(07)
                $conteudo .= self::complementoRegistro(7, "zeros");                 //11  Uso exclusivo da banco                          066 071        9(07)
            }

        } else {
            
            if ($emp['cd_banco'] != Banco::BANCO_RIBEIRAO_PRETO) {
                $conteudo .= self::limit($emp['conta_corrente'], 12, "zeros");      //10a Conta corrente                                  059 070        9(12)
            } 

            if ($emp['cd_banco'] == '341') { //Banco Itau
                $conteudo .= self::complementoRegistro(1, "brancos");           //10b Complemento de registro                         071 071        X(01)
            } elseif ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO) {
                $bancoRibeiraoPreto .= $emp['cd_cedente'];
            } else {
                $conteudo .= self::limit($emp['conta_corrente_dv'], 1, "zeros");//10b Digito verificador da Conta corrente      071 071        9(01)
            }

            if ($cdBanco == '237' || $cdBanco == '004')                            //Condicional do bradesco, banco nordeste
                $conteudo .= self::complementoRegistro(1, "brancos");               //10c Digito do banco (padrao brancos bradesco)       072 072        X(01)
            elseif ($cdBanco == '341')
                $conteudo .= self::limit($emp['conta_corrente_dv'], 1, "brancos"); //10c Digito do banco (padrao brancos itau)          072 072        X(01)
            elseif ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO)
                $bancoRibeiraoPreto .= '';
            else
                $conteudo .= self::complementoRegistro(1, "zeros");             //10c Digito do banco (padrao 0)                      072 072        9(01)
        }
        if ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO) {
            $conteudo .= self::limit('050' . $bancoRibeiraoPreto, 20, "brancos");  // Conta Corrente (posio 53 a 72) junta as informaes contidas na variavel $bancoRibeiraoPreto
        }

        $conteudo .= self::limit(Helper::removeCaracteresEspeciais($empresa), 30);                                         //12  Nome da construtora                      073 102        X(30)
        
        if ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO)
            $conteudo .= strtoupper(self::limit(Helper::removeCaracteresEspeciais($nmBanco) . ' S/A', 30));                    //13  Nome do banco, para ribeira tem que ser em maiusculo e com S/A no final                            103 132        X(30)
        else
            $conteudo .= self::limit(Helper::removeCaracteresEspeciais($nmBanco), 30);                                         //13  Nome do banco                            103 132        X(30)
        
        $conteudo .= self::complementoRegistro(10, "brancos");                          //14  Uso exclusivo da banco                133 142        9(10)
        $conteudo .= '1';                                                               //15  Codigo 1 = remessa; 2 = retorno                143 143        9(01)
        $conteudo .= date("dmY");                                                       //16  Data de geracao                        144 151        9(08)
        $conteudo .= date("His");                                                       //17  Hora de geracao de geracao            152 157        9(06)
        $conteudo .= self::limit($nrRemessa, 6, 'zeros');                               //18  Numero sequencial do arquivo       158 163        9(06)
        
        if ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO)
            $conteudo .= '101';                                                         //19  Layout do arquivo                               164 166        9(03)
        else
            $conteudo .= self::versionamentoLayout($cdBanco, $emp['cd_cedente'])['header']; //19  Layout do arquivo                               164 166        9(03)
        
        if ($cdBanco == '237') {
            $conteudo .= '06250';                                                       //20  Densidade de gravacao do arquivo                167 171        9(05)
        } elseif ($cdBanco == '748' || $cdBanco == '741') {
            $conteudo .= '01600';                                                       //20  Densidade de gravacao do arquivo                167 171        9(05)
        } else {
            $conteudo .= '00000';                                                       //20  Densidade de gravacao do arquivo                167 171        9(05)
        }
        $conteudo .= self::complementoRegistro(20, "brancos");                          //21  Reservado banco                       172 191        9(20)
        $conteudo .= self::limit(($cdBanco == '104') ? 'REMESSA-PRODUCAO' : '', 20, "brancos", false, false); //22  Tipo de remessa    192 211        9(20)
        $conteudo .= self::complementoRegistro(14, "brancos");                          //23  Uso do fenabran                       212 225        X(14)
        if ($cdBanco == '341')
            $conteudo .= self::complementoRegistro(3, "zeros");                      //23  Uso do fenabran                          226 228        9(03)
        else
            $conteudo .= self::complementoRegistro(3, "brancos");                    //23  Uso do fenabran                          226 228        X(03)
        $conteudo .= self::complementoRegistro(12, "brancos");                       //23  Uso do fenabran                          229 240        X(12)
        $conteudo .= chr(13) . chr(10);                                                 //quebra de linha

        return $conteudo;
    }

    /**
     * Gera o Trailer do arquivo
     * @param $cdBanco
     * @param $nrLote
     * @param $nrRegistros
     * @return string
     */
    public static function gerarTrailer($cdBanco, $nrLote, $nrRegistros)
    {
        ##  REGISTRO TRAILER DO ARQUIVO
        #   NOME DO CAMPO                                                                       #SIGNIFICADO                                           #POSICAO    #TAMANHO
        $conteudo = $cdBanco;                                                                   //  Codigo do Banco para compensacao (104 = CAIXA)      001 003     9(03)
        $conteudo .= "9999";                                                                    //  Lote de servico (9999 = trailer do arquivo)         004 007     9(04)
        $conteudo .= "9";                                                                       //  tipo registro (9 = trailer)                         008 008     9(01)
        $conteudo .= self::complementoRegistro(9, "brancos");                                   //  USO FEBRABAN                                        009 017     X(06)
        $conteudo .= self::limit($nrLote, 6, "zeros");                                          //  Quantidade de lotes                                 018 023     X(06)
        $conteudo .= self::limit($nrRegistros + 2, 6, "zeros");                                 //  Quantidade de Registros                             024 029     X(06)
        $conteudo .= self::complementoRegistro(6, "zeros");                                     //  Qtde de Contas p/ Conc. (Lotes)                     030 035     X(06)
        $conteudo .= self::complementoRegistro(205, "brancos");                                 //  USO FEBRABAN                                        036 240     X(205)
        $conteudo .= chr(13) . chr(10);                                                         //  quebra de linha

        return $conteudo;
    }

    /**
     * Gera o header do lote apenas
     * @param $emp
     * @param $nrArquivo
     * @return string
     */
    public static function geraHeaderLote($emp, $nrArquivo)
    {
        $empresa = strtoupper(self::retirar_caracteres_especiais($emp["nm_cedente"]));
        $cnpjCpfCedente = ($emp["cnpj_cedente"])?$emp["cnpj_cedente"]:$emp["cpf_cedente"];

        //apenas para caixa verifica se e pessoa fisica ou juridica e seta 1 ou 2, para outros bancos, seta 2
        $tpInscricao = '2';
        if ($emp['cd_banco'] == '104' || $emp['cd_banco'] == '756'  || $emp['cd_banco'] == Banco::BANCO_BRADESCO || $emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO || $emp['cd_banco'] == Banco::BANCO_DO_BRASIL || $emp['cd_banco'] == Banco::BANCO_SAFRA || $emp['cd_banco'] == Banco::BANCO_ITAU || $emp['cd_banco'] == Banco::BANCO_NORDESTE || $emp['cd_banco'] == Banco::BANCO_SANTANDER || $emp['cd_banco'] == Banco::BANCO_SICREDI) { //Registro complementar para caixa economica
            $tpInscricao = ($emp["cnpj_cedente"])?'2':'1';
        }

        $codEmpresa = str_replace(array(".", "/", "-"), "", $cnpjCpfCedente);

        ##  HEADER DO LOTE (OBRIGATORIO) - REGISTRO 1
        #  NOME DO CAMPO                                                                #SIGNIFICADO                                            #POSICAO    #TAMANHO
        $conteudo = $emp['cd_banco'];                                                   //  Codigo banco                                        001 003     9(03)
        $conteudo .= self::sequencial($nrArquivo, 4);                                   //  Lote                   Lote de servico              004 007     9(04)
        $conteudo .= '1';                                                               //  Registro header do lote                             008 008     9(01)
        $conteudo .= 'R';                                                               //  Tipo de operacao                                    009 009     X(01)
        $conteudo .= '01';                                                          //  Tipo de servico                                     010 011     9(02)

        if ($emp['cd_banco'] == '237' || $emp['cd_banco'] == '741' || $emp['cd_banco'] == '756' || $emp['cd_banco'] == Banco::BANCO_BANPARA || $emp['cd_banco'] == Banco::BANCO_DO_BRASIL) {
            $conteudo .= '  ';                                                          //  Uso exclusivo Febraban                              012 013     X(02)
        } else {
            $conteudo .= '00';                                                          //  Uso exclusivo Febraban                              012 013     9(02)
        }
        $conteudo .= self::versionamentoLayout($emp['cd_banco'], $emp['cd_cedente'])['lote'];               //  nr da versao do layout do lote                      014 016     9(03)
        $conteudo .= ' ';                                                               //  Uso exclusivo do fenabran                           017 017     X(01)
        $conteudo .= $tpInscricao;                                                      //  Tipo de inscricao do empreendimento 2 = CNPJ        018 018     9(01)

        $conteudo .= self::limit($codEmpresa, 15, "zeros");                             //  nr de inscricao do empreendimento                   019 033     9(15)

        if ($emp['cd_banco'] == '104') { //Registro complementar para caixa econimica
            
            if (strlen($emp['cd_cedente']) < 7) {
                $conteudo .= self::limit($emp['cd_cedente'], 6, "zeros");                   //  11.1 Codigo do cedente do empreendimento                 034 039     9(06)
                $conteudo .= self::complementoRegistro(14, "zeros");                        //  11.1A Uso exclusivo da caixa                              040 053     9(14)
            } else {
                $conteudo .= self::limit($emp['cd_cedente'], 7, "zeros");                   //  11.1 Codigo do cedente do empreendimento                 034 039     9(06)
                $conteudo .= self::complementoRegistro(13, "zeros");                        //  11.1A Uso exclusivo da caixa                              040 053     9(14)
            }
        } elseif ($emp['cd_banco'] == '001') {
            $conteudo .= self::limit($emp['cd_cedente'], 9, "zeros");                   //  Codigo do cedente do empreendimento                 034 042     9(09)
            $conteudo .= '0014';                                                        //  Codigo 0014 para cobranca cedente                   043 046     9(04)
            $conteudo .= self::limit($emp['carteira'], 2, "zeros");                     //  Codigo da carteira                                  047 048     9(02)
            $conteudo .= self::limit($emp['carteira_variacao'], 3, "zeros");            //  Codigo de variacao da carteira (019 padrao)         049 051     9(03)
            $conteudo .= self::complementoRegistro(2, "brancos");                       //  Uso exclusivo do banco                              052 053     9(02)
        } elseif ($emp['cd_banco'] == '237') { // Bradesco
            $conteudo .= self::limit($emp['cd_cedente'], 20, "zeros", true);            //  Codigo do convenio do cedente                       034 053     9(20)
        } elseif ($emp['cd_banco'] == Banco::BANCO_SAFRA) {
            $conteudo .= self::limit($emp['cd_cedente'], 20, "zeros");           //  Codigo do convenio do cedente                       034 053     9(20)
        } elseif ($emp['cd_banco'] == '004') {                                        // Banco do Nordeste
            $conteudo .= self::complementoRegistro(20, "zeros");                        // Uso exclusivo do banco                      034 053     9(20)
        } elseif ($emp['cd_banco'] == Banco::BANCO_BANPARA){
            $conteudo .= self::limit($emp['cd_cedente'], 20, "brancos");          //07  Codigo do convenio do cedente
        } elseif ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO) {
            $conteudo .= self::limit('050' . $emp['agencia'] . $emp['agencia_dv'] . $emp['cd_cedente'], 20, "brancos");        //  Uso exclusivo do banco                              034 053     X(20)
        } else {
            $conteudo .= self::complementoRegistro(20, "brancos");                      //  Uso exclusivo do banco                              034 053     X(20)
        }

        $bancoRibeiraoPreto = '';

        if ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO) {
            $bancoRibeiraoPreto .= $emp["agencia"];
        } else {
            $conteudo .= self::limit($emp["agencia"], 5, "zeros");                          //  Agencia                                             054 058     9(05)
        }

        if ($emp['cd_banco'] == '341')  //Banco Itau
            $conteudo .= self::complementoRegistro(1, "brancos");                       // Complemento de registro                              059 059        X(01)
        elseif ($emp['cd_banco'] == '748')  //Banco Sicredi
            $conteudo .= ($emp["agencia_dv"])?$emp["agencia_dv"]:' ';                   //  Digito verificador da agencia                       059 059     9(01)
        elseif ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO)
            $bancoRibeiraoPreto .= $emp["agencia_dv"];
        else
            $conteudo .= self::limit($emp["agencia_dv"], 1, "zeros");                   //  Digito verificador da agencia                       059 059     9(01)
        
        if ($emp['cd_banco'] == '104') { //Codicional se a caixa economica incluindo registro especifico
            
            if (strlen($emp['cd_cedente']) < 7) {
                $conteudo .= self::limit($emp['cd_cedente'], 6, "zeros");                   //  Codigo do convenio                                  060 065     9(06)
                $conteudo .= self::complementoRegistro(8, "zeros");                         //  Uso exclusivo da banco                              066 073     9(08)
            } else {
                // $conteudo .= self::limit($emp['cd_cedente'], 7, "zeros");                   //  Codigo do convenio                                  060 065     9(06)
                $conteudo .= self::complementoRegistro(7, "zeros");                   //  Codigo do convenio                                  060 065     9(06)
                $conteudo .= self::complementoRegistro(7, "zeros");                         //  Uso exclusivo da banco                              066 073     9(08)
            }

        } else {

            if ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO)
                $bancoRibeiraoPreto .= $emp['cd_cedente'];
            else
                $conteudo .= self::limit($emp['conta_corrente'], 12, "zeros");              // Conta corrente                                       060 071     9(12)
            
            if ($emp['cd_banco'] == '341') //Banco Itau
                $conteudo .= self::complementoRegistro(1, "brancos");         // Complemento de registro                         072 072        X(01)
            elseif ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO)
                $bancoRibeiraoPreto .= '';
            else
                $conteudo .= self::limit($emp['conta_corrente_dv'], 1, "zeros");            // Digito verificador da Conta corrente                 072 072     9(01)
            
            if ($emp['cd_banco'] == '237' || $emp['cd_banco'] == '756' || $emp['cd_banco'] == '004') //Condicional do bradesco/Sicoob/b. nordeste
                $conteudo .= self::complementoRegistro(1, "brancos");                   // Digito do banco (padrao brancos bradesco)            073 073     X(01)
            elseif ($emp['cd_banco'] == '341')
                $conteudo .= self::limit($emp['conta_corrente_dv'], 1, "brancos"); // Digito do banco (padrao brancos itau)          073 073        X(01)
            elseif ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO)
                $bancoRibeiraoPreto .= '';
            else
                $conteudo .= self::complementoRegistro(1, "zeros");                     // Digito do banco (padrao 0)                           073 073     9(01)
        }

        if ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO) {
            $conteudo .= self::limit('050' . $bancoRibeiraoPreto, 20, "brancos");
        }
        
        $conteudo .= self::limit(Helper::removeCaracteresEspeciais($empresa), 30, "brancos");                              //  Nome da empresa                                     074 103     9(30)
        $conteudo .= self::complementoRegistro(80, "brancos");                          //  Caracteres extras da mensagem                       104 183     9(80)
        $conteudo .= self::limit($emp['idRemessa'], 8, "zeros");                        //  Numero de remessa (sequencial)                      184 191     9(08)
        $conteudo .= date("dmY");                                                       //  Data de gravacao da remessa                         192 199     9(08)
        $conteudo .= self::complementoRegistro(8, "zeros");                             //  Data do credito (apenas para retorno)               200 207     9(08)
        $conteudo .= self::complementoRegistro(33, "brancos");                          //  Uso exclusivo do CNAB                               208 240     9(33)
        $conteudo .= chr(13) . chr(10);                                                 //  quebra de linha

        return $conteudo;
    }

    /**
     * Gera trailer do lote apenas
     * @param $sequencial
     * @param $emp
     * @param $parcelas
     * @param $nrArquivo
     * @param $vl_total
     * @return string
     */
    public static function gerarTrailerLote($sequencial, $emp, $parcelas, $nrArquivo, $vl_total)
    {
        ##  TRAILER DO LOTE (OBRIGATORIO) - REGISTRO 5
        #  NOME DO CAMPO                                                                #SIGNIFICADO                                            #POSICAO    #TAMANHO
        $conteudo = $emp['cd_banco'];                                                   //  Codigo banco                                        001 003     9(03)
        $conteudo .= self::sequencial($nrArquivo, 4);                                   //  Lote                   Lote de servico            004 007     9(04)
        $conteudo .= '5';                                                               //  Registro trailer do lote                            008 008     9(01)
        $conteudo .= self::complementoRegistro(9, "brancos");                           //  Filler                                              009 017     X(09)
        
        if ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO)
            $conteudo .= self::complementoRegistro(6, "zeros");                              //  Numero de registros no lote                        018 023     9(06)
        else
            $conteudo .= self::limit($sequencial, 6, "zeros");                              //  Numero de registros no lote                        018 023     9(06)
        
        if ($emp['cd_banco'] == '237' || $emp['cd_banco'] == '741' || $emp['cd_banco'] == '748') { //Condicional do bradesco, banco ribeiro preto e sicredi
            $conteudo .= self::complementoRegistro(6, "zeros");                         //  Numero de titulos no lote                          024 029     9(06)
        } else
            $conteudo .= self::limit(count($parcelas), 6, "zeros");                     //  Numero de titulos no lote                          024 029     9(06)
        
        if ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO)
            $conteudo .= self::complementoRegistro(17, "zeros");
        else
            $conteudo .= self::limit($vl_total, 17, "zeros");                               //  Valor total dos titulos no lote                     030 046     9(17)
        
        $conteudo .= self::complementoRegistro(6, "zeros");                             //  Numero de titulos caucionados no lote              047 052     9(06)
        $conteudo .= self::complementoRegistro(17, "zeros");                            //  Valor total dos titulos caucionados no lote         053 069     9(17)
        $conteudo .= self::complementoRegistro(6, "zeros");                             //  Numero de titulos descontados no lote              070 075     9(06)
        $conteudo .= self::complementoRegistro(17, "zeros");                            //  Valor total dos titulos descontados no lote         076 092     9(17)
        if ($emp['cd_banco'] == '237' || $emp['cd_banco'] == '756') { //Condicional do bradesco /banco ribeiro preto/Banco Sicob
            $conteudo .= self::complementoRegistro(6, "zeros");                         //  Uso exclusivo do banco                              093 098     9(06)
            $conteudo .= self::complementoRegistro(17, "zeros");                        //  Uso exclusivo do banco                              099 115     9(17)
            $conteudo .= self::complementoRegistro(8, "zeros");                         //  Uso exclusivo do banco                              116 123     9(08)
        } elseif ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO) {
            $conteudo .= self::complementoRegistro(6, "zeros");                         //  Uso exclusivo do banco                              093 098     9(06)
            $conteudo .= self::complementoRegistro(17, "zeros");                        //  Uso exclusivo do banco                              099 115     9(17)
            $conteudo .= self::complementoRegistro(8, "brancos");                         //  Uso exclusivo do banco                              116 123     9(08)
        } 
        else
            $conteudo .= self::complementoRegistro(31, "brancos");                      //  Uso exclusivo do CNAB                               093 123     9(31)
        $conteudo .= self::complementoRegistro(117, "brancos");                         //  Uso exclusivo do CNAB                               124 240     9(117)
        $conteudo .= chr(13) . chr(10);                                                 //  essa e a quebra de linha

        return $conteudo;
    }

    /**
     * Gera apenas um registro de parcela e retorna o conteudo
     * @param $codMovimento
     * @param $emp
     * @param $parcela
     * @param $nrArquivo
     * @param $idContrato
     * @param $sequencial
     * @return string
     */
    public static function gerarDetalheApenas($codMovimento, $emp, $parcela, $nrArquivo, $idContrato, $sequencial, $emissor = "2")
    {
        $contrato = Contrato::findContratoById($idContrato);

        //Dados para geracao
        $registro = 1;
        $cpfCnpj = in_array($contrato["tp_pessoa"], array('F', 'C')) ? str_replace(array(".", "/", "-"), "", $contrato["nr_cpf"]) : str_replace(array(".", "/", "-"), "", $contrato["nr_cnpj"]);
        $tipoRegistro = in_array($contrato["tp_pessoa"], array('F', 'C')) ? '1' : '2';
        $codCarteira = self::verificaCodigoCarteira($emp['cd_banco'], $emp['carteira']);

        //Reduz do valor da parcela o desconto cadastrado
        $vl_parcela = str_pad(str_replace(array(".", ","), "", number_format($parcela["vl_parcela"], 2)), 15, '0', STR_PAD_LEFT);
        $vlParcela1 = $parcela["vl_parcela"];

        if ($emp["tp_juros"] != 'S') {
            $juros = ($emp["tp_juros"] == 'P') ? (float)$emp["juros"] * (float)$parcela["vl_parcela"] / 100 : $emp["juros"];
        } else {
            $juros = 0;
        }
        $juros = str_replace(array(".", ","), "", number_format($juros, 2));
        $multa = str_replace(array(".", ","), "", number_format($emp["multa"], 2));

        if ($emp['cd_banco'] == '741')
            $multa = str_replace(array(".", ","), "", number_format($emp["multa"], 4));

        $nossoNumero = self::geraNossoNumero($emp['cd_banco'], $parcela, $emp);
        
        $dias_permanencia = (isset($emp['qt_dias_permanencia']) && (int)$emp['qt_dias_permanencia'] > 5) ? $emp['qt_dias_permanencia'] : 90;

        $contrato['tp_parcela'] = $parcela['tp_parcela'];
        $contrato['dt_parcela'] = $parcela['dt_parcela'];        

        $descontoParcela = Parcela::getDescontoParcela($contrato, $vlParcela1);     
            
        if((float)$parcela["vl_parcela_2avia"] > 0 && !$parcela['desconto_2via'])
            $descontoParcela['vlDesconto'] = 0;
        $vlDesconto = str_pad(str_replace(array(".", ","), "", number_format($descontoParcela['vlDesconto'], 2)), 15, '0', STR_PAD_LEFT);
        

        ## SEGMENTO P (OBRIGATORIO)                                                     ##  NOME DO CAMPO                                       POSICAO    TAMANHO
        $conteudo = $emp['cd_banco'];                                                   //  Codigo do banco                                     001 003     9(03)
        $conteudo .= self::sequencial($nrArquivo, 4);                                   //  Lote do servico                                     004 007     9(04)
        $conteudo .= '3';                                                               //  Registro detalhe                                    008 008     9(01)
        $conteudo .= self::sequencial($sequencial, 5);                                  //  Numero sequencial do registro no lote               009 013     9(05)
        $conteudo .= 'P';                                                               //  Codigo do segmento                                  014 014     X(01)
        $conteudo .= ' ';                                                               //  Uso do banco                                        015 015     X(01)
        $conteudo .= $codMovimento;                                                     //  Codigo de movimentivao (Informado ao chamar funcao) 016 017     9(02)
        
        $bancoRibeiraoPreto = '';
        
        if ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO)
            $bancoRibeiraoPreto .= $emp["agencia"];                                        //  Agencia                                             018 022     9(05)
        else 
            $conteudo .= self::limit($emp["agencia"], 5, "zeros");                          //  Agencia                                             018 022     9(05)
        
        if ($emp['cd_banco'] == '341')  //Banco Itau
            $conteudo .= self::complementoRegistro(1, "brancos");                       // Complemento de registro                          023 023        X(01)
        elseif ($emp['cd_banco'] == '748')  //Banco Sicredi
            $conteudo .= ($emp["agencia_dv"])?$emp["agencia_dv"]:' ';                   //  Dv da agencia                                       023 023     9(01)
        elseif ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO)
            $bancoRibeiraoPreto .= $emp["agencia_dv"];                                  //  Dv da agencia                                       023 023     9(01)
        else
            $conteudo .= self::limit($emp["agencia_dv"], 1, "zeros");                   //  Dv da agencia                                       023 023     9(01)
        
        if ($emp['cd_banco'] == '104') { //Codicional se e caixa economica incluindo registro especifico
            
            if (strlen($emp['cd_cedente']) < 7) {
                $conteudo .= self::limit($emp['cd_cedente'], 6, "zeros");                   //  10.3P Codigo do convenio do cedente                       024 029     9(06)
                $conteudo .= self::complementoRegistro(8, "zeros");                         //  11.3P Uso exclusivo da caixa                              030 037     9(08)
            } else {
                $conteudo .= self::limit($emp['cd_cedente'], 7, "zeros");                   //  10.3P Codigo do convenio do cedente                       024 029     9(07)
                $conteudo .= self::complementoRegistro(7, "zeros");                         //  11.3P Uso exclusivo da caixa                              030 037     9(07)
            }
        } else {

            if ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO)
                $bancoRibeiraoPreto .= $emp['cd_cedente'];                                  //  Conta corrente                                      024 035     9(12)
            else
                $conteudo .= self::limit($emp['conta_corrente'], 12, "zeros");              //  Conta corrente                                      024 035     9(12)
            
            if ($emp['cd_banco'] == '341')  //Banco Itau
                $conteudo .= self::complementoRegistro(1, "brancos");                   // Complemento de registro                          036 036        X(01)
            elseif ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO)
                $bancoRibeiraoPreto .= '';
            else
                $conteudo .= self::limit($emp['conta_corrente_dv'], 1, "zeros");        //  Digito verificador da Conta corrente                036 036     9(01)
            
            if ($emp['cd_banco'] == '237' || $emp['cd_banco'] == '756' || $emp['cd_banco'] == '004')
                $conteudo .= self::complementoRegistro(1, "brancos");                   //  Digito do banco (padrco ' ')                        037 037     9(01)
            elseif ($emp['cd_banco'] == '341')
                $conteudo .= self::limit($emp['conta_corrente_dv'], 1, "brancos");      // Digito do banco (padrao brancos itau)          037 037        X(01)
            elseif ($emp['cd_banco'] == Banco::BANCO_BANPARA)                                       //  Digito do banco                        037 037     9(01)
                $conteudo .= self::limit($emp["agencia_dv"], 1, "zeros");
            elseif ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO)
                $bancoRibeiraoPreto .= '';
            else
                $conteudo .= self::complementoRegistro(1, "zeros");                     //  Digito do banco (padrao 0)                          037 037     9(01)
        }

        if ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO)
            $conteudo .= self::limit('050' . $bancoRibeiraoPreto, 20, "brancos");          //junta informaes do banco ribeiro preto para formar a C/C

        if ($emp['cd_banco'] == '341') {
            $conteudo .= self::limit($nossoNumero, 20, "zeros");                        //  Identific. do titulo no banco                       038 057     9(20)
            $conteudo .= self::complementoRegistro(5, 'zeros');                         //  Cd da carteira 1= cob. simples. 2= cob. caucionada  058 062     9(01)
            if ($parcela['nr_boleto']) {
                $conteudo .= self::limit($parcela["nr_boleto"], 10, "zeros");     //  Numero do documento de cobranca                     063 072     9(10)
            } else {
                $conteudo .= self::limit($parcela["id_parcela_contrato"], 10, "zeros");     //  Numero do documento de cobranca                     063 072     9(10)
            }
            
            $conteudo .= self::complementoRegistro(5, "brancos");                       //  Reservado para uso do banco                         073 077     X(05)
        } else {
            
            if ($emp['cd_banco'] == Banco::BANCO_SAFRA) {                                        //  Identific. do titulo no banco                       038 057     9(20)
                $conteudo .= substr($nossoNumero, -9);
                $conteudo .= self::complementoRegistro(11, "brancos");
            } elseif ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO) {
                $conteudo .= '5';                                                           // Servio de Classificao (posio 38): informar ?5? (SOMENTE BANCO RIBEIRAO)
                $conteudo .= self::complementoRegistro(5, 'zeros');                        //  Modalidade com correspondente: informar ?000? ;      39 a 43     9(19)
                $conteudo .= '110';                                                        //Modalidade com banco cedente informar ?112?;           44 a 46
                
                $dvRibeiraoPreto = self::digitoVerificadorRibeiraoPreto($emp['agencia'], $emp['carteira'], $parcela['nr_boleto']);

                $conteudo .= self::limit($parcela['nr_boleto'].$dvRibeiraoPreto, 11, "zeros");                        //  Identific. do titulo no banco                       047 057     9(19)
            } else {
                $conteudo .= self::limit($nossoNumero, 20, "zeros");                        //  Identific. do titulo no banco                       038 057     9(20)
            }

            $conteudo .= $emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO ? '1' : $codCarteira;                                                  //  Cd da carteira 1= cob. simples. 2= cob. caucionada  058 058     9(01)
            
            if ($emp['cd_banco'] == '756') { //Sicoob
                $conteudo .= '0';                                                       //  Forma de cadastramento do titulo no banco           059 059     9(01)
                $conteudo .= ' ';                                                       //  Tipo de documento                                   060 060     9(01)
            } elseif ($emp['cd_banco'] == Banco::BANCO_BANPARA){
                $conteudo .= '1';                                                       //  Tipo de documento  060 060     9(01)
                $conteudo .= '1';                                                       //  Tipo de documento  060 060     9(01)
            } elseif ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO) {
                $conteudo .= '1';                                                       //  Tipo de documento  059 059     9(01)
                $conteudo .= '1';                                                       //  Tipo de documento  060 060     9(01)
            } else {
                $conteudo .= $registro;                                                 //  Forma de cadastramento do titulo no banco           059 059     9(01)
                $conteudo .= '2';                                                       //  Tipo de documento                                   060 060     9(01)
            }
            
            $conteudo .= $emissor;                                                           //  Emissao do bloqueto 2                061 061     9(01)                                                           //  Emissao do bloqueto 2 = cliente emite               061 061     9(01)
            
            if ($emp['cd_banco'] == '104') {
                $conteudo .= '0';   //  Identificacao da distribuicao 1 = Banco distribui   062 062     9(01)
            }
            else {
                $conteudo .= $emissor;   //  Identificacao da distribuicao 1 = Banco distribui   062 062     9(01)
            }
            
            if ($emp['cd_banco'] == '756' || $emp['cd_banco'] == '004') { //Sicoob ou Banco Nordeste
                $conteudo .= self::limit($parcela["id_parcela_contrato"], 15, "zeros");     //  Numero do documento de cobranca                     063 077     9(11)
            }  elseif ($emp['cd_banco'] == Banco::BANCO_SAFRA){
//                $conteudo .= substr($nossoNumero, -9); //  Numero do documento de cobranca                     063 073     9(11)
                $conteudo .= self::limit($parcela["id_parcela_contrato"], 10, "zeros");
                $conteudo .= self::complementoRegistro(6, "brancos");//  Numero do documento de cobranca
            } else {
                if ($emp['cd_banco'] == '741'){
                    $conteudo .= self::limit($parcela["nr_boleto"], 11, "zeros");     //  Numero do documento de cobranca                     063 073     9(11)    
                } else {
                    if ($parcela['nr_boleto']) {
                        $conteudo .= self::limit($parcela["nr_boleto"], 11, "zeros");     //  Numero do documento de cobranca                     063 073     9(11)
                    } else {
                        $conteudo .= self::limit($parcela["id_parcela_contrato"], 11, "zeros");     //  Numero do documento de cobranca                     063 073     9(11)
                    }
                }
                $conteudo .= self::complementoRegistro(4, "brancos");                       //  Reservado para uso do banco                         074 077     9(04)
            }
        }
        $conteudo .= date("dmY", strtotime($parcela["dt_parcela"]));                    //  Vencimento da prestacao                             078 085     9(08)
        $conteudo .= $vl_parcela;                                                       //  Valor da prestaco                                  086 100     9(15)
        $conteudo .= self::complementoRegistro(5, "zeros");                             //  Agencia responsavel pela cobranca (USO DA CAIXA)    101 105     9(05)
        if ($emp['cd_banco'] == '756' || $emp['cd_banco'] == '741') { //Condicional do Sicoo e Ribeiro Preto
            $conteudo .= ' ';                                                           //  Digito verificador da agencia (USO DA CAIXA)        106 106     9(01)
        } else {
            $conteudo .= self::complementoRegistro(1, "zeros");                         //  Digito verificador da agencia (USO DA CAIXA)        106 106     9(01)
        }
        if ($emp['cd_banco'] == '001' || $emp['cd_banco'] == '756' || $emp['cd_banco'] == '004' || $emp['cd_banco'] == Banco::BANCO_BRADESCO || $emp['cd_banco'] == Banco::BANCO_CAIXA_ECONOMICA || $emp['cd_banco'] == Banco::BANCO_BANPARA) {  //  Especie do titulo (99 = outros)                     107 108     9(02)
            $conteudo .= '02';                                                          //  Especie do titulo (99 = outros)                     107 108     9(02)
        } elseif ($emp['cd_banco'] == '748' && $emp['especie_documento'] == 'DMI') {
            $conteudo .= '03';                                                          //  SOMENTE PARA O BANCO SICREDI. Especie do titulo (03 = DMI - Duplicata mercantil por indicao)                     107 108     9(02)
        } else {
            $conteudo .= '99';                                                          //  Especie do titulo (99 = outros)                     107 108     9(02)
        }
        $conteudo .= "N";                                                               //  Aceite A = Aceite N = Nao aceite                    109 109     X(01)
        $conteudo .= date("dmY");                                                       //  Data da emissao da parcela                          110 117     9(08)
        $conteudo .= ($emp['cd_banco'] == '341') ? "0" : "1";                           //  Codigo do juros/mora (1 = diario)                   118 118     9(01)
        if (in_array($emp['cd_banco'], ['104', '422'])) { //Condicional do Sicoob e caixa
            $conteudo .= date("dmY", strtotime($parcela["dt_parcela"]));                //  Data do juros. branco = data vencimento             119 126     9(08)
        } elseif ($emp['cd_banco'] == '756' || $emp['cd_banco'] == '741') {
            $conteudo .= date("dmY", strtotime($parcela["dt_parcela"] . ' + 1 days'));  // Data do juros, do Sicoob e Ribeiro Preto + 1 dia                     119 126
        } else {
            $conteudo .= self::complementoRegistro(8, "zeros");                         //  Data do juros. branco = data vencimento             119 126     9(08)
        }
        $conteudo .= self::limit($juros, 15, "zeros");                                  //  Juros por dia                                       127 141     9(13)
        if ($descontoParcela['flDesconto'] != 'S' && (float)$descontoParcela['vlDesconto'] > 0) {
            $conteudo .= "1";                                                           //  Codigo do desconto. 1 = valor 2 = percentual        142 142     9(01)
            $conteudo .= date("dmY", strtotime($parcela["dt_parcela"]));   
        } 
        else {
            $conteudo .= "0";                                                           //  Codigo do desconto. 1 = valor 2 = percentual        142 142     9(01)
            $conteudo .= self::complementoRegistro(8, "zeros");                         //  Data do desconto                                    143 150     9(08)
        }
        $conteudo .= $vlDesconto;                                                       //  Valor do desconto                                   151 165     9(13)
        $conteudo .= self::complementoRegistro(15, "zeros");                            //  Valor IOF a pagar                                   166 180     9(13)
        $conteudo .= self::complementoRegistro(15, "zeros");                            //  Valor do abatimento                                 181 195     9(13)

        if ($emp['cd_banco'] == '741') {
            $conteudo .= self::limit($parcela["nr_boleto"], 25, "zeros");         //  Codigo de identificacao da parcela na empresa       196 220     9(25)    
        } else {
            if($parcela['nr_boleto']) {
                $conteudo .= self::limit($parcela["nr_boleto"], 25, "zeros");         //  Codigo de identificacao da parcela na empresa       196 220     9(25)
            } else {
                $conteudo .= self::limit($parcela["id_parcela_contrato"], 25, "zeros");         //  Codigo de identificacao da parcela na empresa       196 220     9(25)
            }
        }
        $conteudo .= "3";                                                               //  Codigo para protesto (3 = sem protesto)             221 221     9(01)
        $conteudo .= self::complementoRegistro(2, "zeros");                             //  Numero de dias para protesto                        222 223     9(02)
        if ($emp['cd_banco'] == '756') //Condicional do Sicoob
            $conteudo .= "0";                                                           //  Codigo de Baixa do boleto (2 = nao baixar)          224 224     9(01)
        else
            $conteudo .= "1";                                                           //  Codigo de Baixa do boleto (2 = nao baixar)          224 224     9(01)
        if ($emp['cd_banco'] == '341') {
            $conteudo .= self::limit($dias_permanencia, 2, "zeros");                    //  Numero de dias para baixa                           225 226     9(02)
            $conteudo .= '0';                                                           //  Numero de dias para baixa                           227 227     9(01)
        } elseif ($emp['cd_banco'] == '756') {
            $conteudo .= '   ';                                                         //  Numero de dias para baixa                           225 227     X(03)
        } else {
            $conteudo .= self::limit($dias_permanencia, 3, "zeros");                    //  Numero de dias para baixa                           225 227     X(03)
        }
        $conteudo .= "09";                                                              //  Codigo da moeda (09 = real)                         228 229     9(02)
        $conteudo .= self::complementoRegistro(10, "zeros");                            //  Numero identificador do contrato na empresa         230 239     9(10)
        
        if ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO)
            $conteudo .= "1";                           //  Uso exclusivo do FEBRABAN                           240 240     9(01)
        else
            $conteudo .= self::complementoRegistro(1, "brancos");                           //  Uso exclusivo do FEBRABAN                           240 240     9(01)
        
        $conteudo .= chr(13) . chr(10);                                                 //  essa e a quebra de linha

        $sequencial++;

        $cep = self::retirar_caracteres_especiais($parcela['nr_cep']);
        $nrCep = substr($cep, 0, 5);
        $sufixoCep = substr($cep, 5, 3);

        ## SEGMENTO Q (OBRIGATORIO)                                                         ##  NOME DO CAMPO                                       POSICAO    TAMANHO
        $conteudo .= $emp['cd_banco'];                                                  //  Codigo do banco                                     001 003     9(03)
        $conteudo .= self::sequencial($nrArquivo, 4);                                   //  Lote do servico                                     004 007     9(04)
        $conteudo .= '3';                                                               //  Registro detalhe                                    008 008     9(01)
        $conteudo .= self::sequencial($sequencial, 5);                                  //  Numero sequencial do registro no lote               009 013     9(05)
        $conteudo .= 'Q';                                                               //  Codigo do segmento                                  014 014     X(01)
        $conteudo .= ' ';                                                               //  Uso do banco                                        015 015     X(01)
        $conteudo .= $codMovimento;                                                     //  Codigo de movimentacao (Informado ao chamar funcao) 016 017     9(02)
        $conteudo .= $tipoRegistro;                                                     //  Tipo de inscricao do Sacado                         018 018     9(01)
        $conteudo .= self::limit($cpfCnpj, 15, "zeros");                                //  Numero de inscricao do Sacado                       019 033     9(15)
        $conteudo .= self::limit(Helper::removeCaracteresEspeciais($parcela['nm_pessoa']), 40, 'brancos');                 //  Nome do Sacado                                      034 073     X(40)
        $conteudo .= self::limit(Helper::removeCaracteresEspeciais($parcela['endereco']), 40, 'brancos');                  //  Endereco do Sacado                                  074 113     X(40)
        $conteudo .= self::limit(Helper::removeCaracteresEspeciais($parcela['nm_bairro']), 15, "brancos");                 //  Bairro do Sacado                                    114 128     X(15)
        $conteudo .= self::limit($nrCep, 5, ($emp['cd_banco'] == '341') ? "zeros" : "brancos");//  CEP do Sacado                                       129 133     9(05)
        $conteudo .= self::limit($sufixoCep, 3, ($emp['cd_banco'] == '341') ? "zeros" : "brancos");//  Sufixo do Cep do sacado                             134 136     9(03)
        $conteudo .= self::limit(Helper::removeCaracteresEspeciais($parcela['cidade']), 15, "brancos");                    //  Nome da Cidade do Sacado                            137 151     X(15)
        $conteudo .= self::limit($parcela['uf'], 2, "brancos");                         //  Uf do Sacado                                        152 153     X(02)

        $tipoPagador = '0';                                                             //  Tipo de inscricao do pagador (se diferente)         154 154     9(01)

        if ($emp['cd_banco'] == Banco::BANCO_SICREDI)
            $tipoPagador = $tipoRegistro;                                              //  Tipo de inscricao do pagador (1=Pessoa Fsica; 2=Pessoa Jurdica), no momento essa condio s funciona para o SICREDI         154 154     9(01)

        $conteudo .= $tipoPagador;                                                      

        if ($emp['cd_banco'] == Banco::BANCO_SICREDI) {
            $conteudo .= self::complementoRegistro(15, "brancos");                      //  Numero do documento do pagador (se diferente), SOMENTE PARA O SICREDI       155 169     9(15)
        } else {
            $conteudo .= self::complementoRegistro(15, "zeros");                        //  Numero do documento do pagador (se diferente)       155 169     9(15)
        }
        $conteudo .= self::complementoRegistro(40, "brancos");                          //  Nome do pagador (se diferente)                      170 209     X(40)
        $conteudo .= self::complementoRegistro(3, "zeros");                             //  Codigo do Banco Correspondente na Compensacao       210 212     9(03)
        $conteudo .= self::complementoRegistro(20, "brancos");                          //  Nosso Nr no Banco Correspondente                    213 232     X(20)
        $conteudo .= self::complementoRegistro(8, "brancos");                           //  Reservado para uso do banco                         233 240     X(08)
        $conteudo .= chr(13) . chr(10);                                                 //  essa e a quebra de linha

        $sequencial++;

        ## SEGMENTO R (OPCIONAL)                                                        ##  NOME DO CAMPO                                       POSICAO    TAMANHO
        $conteudo .= $emp['cd_banco'];                                                  //  Codigo do banco                                     001 003     9(03)
        $conteudo .= self::sequencial($nrArquivo, 4);                                   //  Lote do servico                                     004 007     9(04)
        $conteudo .= '3';                                                               //  Registro detalhe                                    008 008     9(01)
        $conteudo .= self::sequencial($sequencial, 5);                                  //  Numero sequencial do registro no lote               009 013     9(05)
        $conteudo .= 'R';                                                               //  Codigo do segmento                                  014 014     X(01)
        $conteudo .= ' ';                                                               //  Uso do banco                                        015 015     X(01)
        $conteudo .= $codMovimento;                                                     //  Codigo de movimentacao (Informado ao chamar funcao) 016 017     9(02)
        $conteudo .= '0';                                                               //  Tipo de Desconto2 do titulo                         018 018     9(01)
        $conteudo .= self::complementoRegistro(8, "zeros");                             //  Data do desconto2 do titulo                         019 026     9(08)
        $conteudo .= self::complementoRegistro(15, "zeros");                            //  Valor do desconto2 do titulo                        027 041     9(13)
        $conteudo .= '0';                                                               //  Tipo de Desconto3 do titulo                         042 042     9(01)
        $conteudo .= self::complementoRegistro(8, "zeros");                             //  Data do desconto3 do titulo                         043 050     9(08)
        $conteudo .= self::complementoRegistro(15, "zeros");                            //  Valor do desconto3 do titulo                        051 065     9(15)
        if ($emp['cd_banco'] == '748') { //Sicredi aceita apenas percentual
            $conteudo .= '2';                                                       //  Tipo de Multa (2 Percentual)                            066 066     9(01)
        } else {
            if ($emp["tp_multa"] == 'P') {
                $conteudo .= '2';                                                   //  Tipo de Multa (0 sem multa/1 vlr fixo/2 Perc)           066 066     9(01)
            } elseif ($emp["tp_multa"] == 'V') {
                $conteudo .= '1';                                                   //  Tipo de Multa (0 sem multa/1 vlr fixo/2 Perc)           066 066     9(01)
            } else {
                $conteudo .= '0';                                                   //  Tipo de Multa (0 sem multa/1 vlr fixo/2 Perc)           066 066     9(01)
            }
        }
        if (in_array($emp['cd_banco'], ['104', '341','001','237', '422'])) {
            $conteudo .= self::limit(date("dmY", strtotime($parcela["dt_parcela"])), 8, "zeros"); //Data da multa do titulo                     067 074     9(08)
        } elseif ($emp['cd_banco'] == '756' || $emp['cd_banco'] == '741') {
            $conteudo .= date("dmY", strtotime($parcela["dt_parcela"] . ' + 1 days'));  //  Data do multa, do Sicoob + 1 dia                    067 074
        } else {
            $conteudo .= self::complementoRegistro(8, "zeros");                         //Data da multa do titulo                               067 074     9(08)
        }
        $conteudo .= self::limit($multa, 15, "zeros");                                  //  Valor da multa do titulo (porcento)                 075 089     9(15)
        $conteudo .= self::complementoRegistro(10, "brancos");                          //  Informacoes do Pagador                              090 099     X(10)
        $conteudo .= self::complementoRegistro(40, "brancos");                          //  Mensagem nr 3                                       100 139     X(40)
        $conteudo .= self::complementoRegistro(40, "brancos");                          //  Mensagem nr 4                                       140 179     X(40)
        if (in_array($emp['cd_banco'], ['237', '741', '341', '756'])) { //Condicional do bradesco, banco ribeiro preto, itau e sicoob
            $conteudo .= self::complementoRegistro(20, "brancos");                      //  Uso Exclusivo FEBRABAN/CNAB                         180 199     X(20)
            $conteudo .= self::complementoRegistro(8, "zeros");                         //  Cod. Ocor. do Pagador                               200 207     9(08)
            $conteudo .= self::complementoRegistro(3, "zeros");                         //  Cod. do Banco na Conta do Debito                    208 210     9(03)
            $conteudo .= self::complementoRegistro(5, "zeros");                         //  Codigo da Agencia do Debito                         211 215     9(05)
            $conteudo .= self::complementoRegistro(1, "brancos");                       //  Digito Verificador da Agencia                       216 216     X(01)
            $conteudo .= self::complementoRegistro(12, "zeros");                        //  Conta Corrente para Dibito                          217 228     9(12)
            $conteudo .= self::complementoRegistro(1, "brancos");                       //  Digito Verificador da Conta                         229 229     X(01)
            $conteudo .= self::complementoRegistro(1, "brancos");                       //  Digito Verificador Ag/Conta                         230 230     X(01)
            if ($emp['cd_banco'] == '237') { //Condicional do bradesco ou banco ribeiro preto
                $conteudo .= '2';                                                       //  Aviso para Dibito Automatico                        231 231     9(01)
            } else {
                $conteudo .= self::complementoRegistro(1, "zeros");                     //  Aviso para Dibito Automatico                        231 231     9(01)
            }
            $conteudo .= self::complementoRegistro(9, "brancos");                       //  Uso Exclusivo FEBRABAN/CNAB                         232 240     X(09)
        } else {
            $conteudo .= self::complementoRegistro(50, "brancos");                      //  E-mail do pagador                                   180 229     X(50)
            $conteudo .= self::complementoRegistro(11, "brancos");                      //  Reservado para uso do banco                         231 240     X(11)
        }
        $conteudo .= chr(13) . chr(10);                                                 //  quebra de linha

        return $conteudo;
    }

    /**
     * * Funcao que gera o registro detalhe utilizando os dados de empreendimento e dos contratos e parcelas
     * @param string $codMovimento
     * @param array $emp
     * @param array $parcelas
     * @param int $nrArquivo
     * @param $idContrato
     * @return mixed
     */
    public static function gerarDetalhe($codMovimento, $emp, $parcelas, $nrArquivo, $idContrato, $emissor = 2)
    {
        $contrato = Contrato::findContratoById($idContrato);

        //Verifica se existem parcelas e contrato
        if (count($parcelas) == 0 || !$contrato)
            return false;

        //Dados para geracao
        $empresa = strtoupper(self::retirar_caracteres_especiais($emp["nm_cedente"]));
        $cnpjCpfCedente = ($emp["cnpj_cedente"])?$emp["cnpj_cedente"]:$emp["cpf_cedente"];

        //apenas para caixa verifica se e pessoa fisica ou juridica e seta 1 ou 2, para outros bancos, seta 2
        $tpInscricao = '2';
        if ($emp['cd_banco'] == '104' || $emp['cd_banco'] == '756') { //Registro complementar para caixa economica
            $tpInscricao = ($emp["cnpj_cedente"])?'2':'1';
        }

        $codEmpresa = str_replace(array(".", "/", "-"), "", $cnpjCpfCedente);
        $cpfCnpj = (in_array($contrato["tp_pessoa"], array('F', 'C'))) ? str_replace(array(".", "/", "-"), "", $contrato["nr_cpf"]) : str_replace(array(".", "/", "-"), "", $contrato["nr_cnpj"]);
        $tipoRegistro = in_array($contrato["tp_pessoa"], array('F', 'C')) ? '1' : '2';
        $codCarteira = self::verificaCodigoCarteira($emp['cd_banco'], $emp['carteira']);
        $dtContrato = str_replace("/", "", Helper::getDate($contrato["dt_contrato"]));

        ##  HEADER DO LOTE (OBRIGATORIO) - REGISTRO 1
        #  NOME DO CAMPO                                                                #SIGNIFICADO                                            #POSICAO    #TAMANHO
        $conteudo = $emp['cd_banco'];                                                   //  Codigo banco                                        001 003     9(03)
        $conteudo .= self::sequencial($nrArquivo, 4);                                   //  Lote                   Lote de servico              004 007     9(04)
        $conteudo .= '1';                                                               //  Registro header do lote                             008 008     9(01)
        $conteudo .= 'R';                                                               //  Tipo de operacao                                    009 009     X(01)
        $conteudo .= '01';                                                          //  Tipo de servico                                     010 011     9(02)
        if (in_array($emp['cd_banco'], ['237', '741', '341', '756', '037'])) {
            $conteudo .= '  ';                                                          //  Uso exclusivo Febraban                              012 013     X(02)
        } else {
            $conteudo .= '00';                                                          //  Uso exclusivo Febraban                              012 013     9(02)
        }
        $conteudo .= self::versionamentoLayout($emp['cd_banco'], $emp['cd_cedente'])['lote'];               //  nr da versao do layout do lote                      014 016     9(03)
        $conteudo .= ' ';                                                               //  Uso exclusivo do fenabran                           017 017     X(01)
        $conteudo .= $tpInscricao;                                                       //  Tipo de inscricao do empreendimento 2 = CNPJ        018 018     9(01)
        $conteudo .= self::limit($codEmpresa, 15, "zeros");                             //  nr de inscricao do empreendimento                   019 033     9(15)
        if ($emp['cd_banco'] == '104') { //Registro complementar para caixa economica
            
            if (strlen($emp['cd_cedente']) < 7) {
                $conteudo .= self::limit($emp['cd_cedente'], 6, "zeros");                   //  11.1 Codigo do cedente do empreendimento                 034 039     9(06)
                $conteudo .= self::complementoRegistro(14, "zeros");                        //  11.1A Uso exclusivo da caixa                              040 053     9(14)
            } else {
                $conteudo .= self::limit($emp['cd_cedente'], 7, "zeros");                   //  11.1 Codigo do cedente do empreendimento                 034 039     9(06)
                $conteudo .= self::complementoRegistro(13, "zeros");                        //  11.1A Uso exclusivo da caixa                              040 053     9(13)
            }

            
        } elseif ($emp['cd_banco'] == '001') {
            $conteudo .= self::limit($emp['cd_cedente'], 9, "zeros");                   //  Codigo do cedente do empreendimento                 034 042     9(09)
            $conteudo .= '0014';                                                        //  Codigo 0014 para cobranca cedente                   043 046     9(04)
            $conteudo .= self::limit($emp['carteira'], 2, "zeros");                     //  Codigo da carteira                                  047 048     9(02)
            $conteudo .= self::limit($emp['carteira_variacao'], 3, "zeros");            //  Codigo de variacao da carteira (019 padrao)         049 051     9(03)
            $conteudo .= self::complementoRegistro(2, "brancos");                       //  Uso exclusivo do banco                              052 053     9(02)
        } elseif ($emp['cd_banco'] == '237' || $emp['cd_banco'] == '741') {
            $conteudo .= self::limit($emp['cd_cedente'], 20, "zeros", true);            //  Codigo do convenio do cedente                       034 053     9(20)
        } elseif ($emp['cd_banco'] == Banco::BANCO_SAFRA) {
            $conteudo .= self::limit($emp['cd_cedente'], 20, "zeros");           //  Codigo do convenio do cedente                       034 053     9(20)
        } elseif ($emp['cd_banco'] == Banco::BANCO_BANPARA) {
            $conteudo .= self::limit($emp['cd_cedente'], 20, "brancos");           //  Codigo do convenio do cedente                       034 053     9(20)
        } elseif ($emp['cd_banco'] == '004') {                                         // Banco do Nordeste
            $conteudo .= self::complementoRegistro(20, "zeros");            // Uso exclusivo do banco                      034 053     9(20)
        } else {
            $conteudo .= self::complementoRegistro(20, "brancos");                      //  Uso exclusivo do banco                              034 053     X(20)
        }

        $conteudo .= self::limit($emp["agencia"], 5, "zeros");                          //  Agencia                                             054 058     9(05)
        if ($emp['cd_banco'] == '341')  //Banco Itau
            $conteudo .= self::complementoRegistro(1, "brancos");                       // Complemento de registro                              059 059     X(01)
        elseif ($emp['cd_banco'] == '748')  //Banco Sicredi
            $conteudo .= ($emp["agencia_dv"])?$emp["agencia_dv"]:' ';                   //  Digito verificador da agencia                       059 059     9(01)
        else
            $conteudo .= self::limit($emp["agencia_dv"], 1, "zeros");                   //  Digito verificador da agencia                       059 059     9(01)

        if ($emp['cd_banco'] == '104') { //Codicional se e caixa economica incluindo registro especifico
            
            if (strlen($emp['cd_cedente']) < 7) {
                $conteudo .= self::limit($emp['cd_cedente'], 6, "zeros");                   //  Codigo do convenio                                  060 065     9(06)
                $conteudo .= self::complementoRegistro(8, "zeros");                         //  Uso exclusivo da banco                              066 073     9(08)
            } else {
                // $conteudo .= self::limit($emp['cd_cedente'], 7, "zeros");                   //  Codigo do convenio                                  060 065     9(06)
                $conteudo .= self::complementoRegistro(7, "zeros");                   //  Codigo do convenio                                  060 065     9(06)
                $conteudo .= self::complementoRegistro(7, "zeros");                         //  Uso exclusivo da banco                              066 073     9(08)
            }

            
        } else {
            $conteudo .= self::limit($emp['conta_corrente'], 12, "zeros");              // Conta corrente                                       060 071     9(12)
            if ($emp['cd_banco'] == '341')  //Banco Itau
                $conteudo .= self::complementoRegistro(1, "brancos");                   // Complemento de registro                              072 072     X(01)
            else
                $conteudo .= self::limit($emp['conta_corrente_dv'], 1, "zeros");        // Digito verificador da Conta corrente                 072 072     9(01)
            if ($emp['cd_banco'] == '237' || $emp['cd_banco'] == '741' || $emp['cd_banco'] == '756')                 //Condicional do bradesco e Sicoob
                $conteudo .= self::complementoRegistro(1, "brancos");                   // Digito do banco (padrao brancos bradesco)            073 073     X(01)
            elseif ($emp['cd_banco'] == '341')
                $conteudo .= self::limit($emp['conta_corrente_dv'], 1, "brancos");      // Digito do banco (padrao brancos itau)                073 073     X(01)
            else
                $conteudo .= self::complementoRegistro(1, "zeros");                     // Digito do banco (padrao 0)                           073 073     9(01)
        }
        $conteudo .= self::limit(Helper::removeCaracteresEspeciais($empresa), 30, "brancos");                              //  Nome da empresa                                     074 103     9(30)
        if ($emp['cd_banco'] == '756') { //Sicoob
            $conteudo .= self::complementoRegistro(80, "brancos");                          //  Mesangem 1 e Mensagem 2                         104 183     9(40)
        } else {
            //Dados da mensagem de impressao (dados do loteamento)
            $conteudo .= (in_array($contrato["tp_pessoa"], array('F', 'C'))) ? '1' : '2';   //  Tipo de pessoa 1 = Fisica 2 = Juridica          104 104     9(01)
            $conteudo .= self::limit($cpfCnpj, 15, "zeros");                                //  Cpf ou cnpj do comprador                        105 119     9(15)
            $conteudo .= self::limit($contrato["quadra"], 10, "brancos");                   //  Quadra                                          120 129     X(10)
            $conteudo .= self::limit($contrato["id_lote"], 10, "brancos");                  //  Numero da unidade                               130 139     X(10)
            $conteudo .= '5';                                                               //  Tipo de imovel 5 = outros                       140 140     9(01)
            $conteudo .= '5';                                                               //  Tipo da unidade 5 = outros                      141 141     X(01)
            $conteudo .= self::complementoRegistro(10, "brancos");                          //  Area util                                       142 151     9(06)
            $conteudo .= self::limit($dtContrato, 8, 'zeros');                              //  Data do contrato                                152 160     9(08)
            $conteudo .= '2';                                                               //  Contrato subrogado 2 = nao                      161 161     9(01)
            $conteudo .= self::complementoRegistro(8, "zeros");                             //  Data da ultima subrogacao                       162 169     9(08)
            $conteudo .= self::limit($contrato["prazo_finan"], 3, "zeros");                 //  Prazo de financiamento total                    170 172     9(03)
            $conteudo .= '07';                                                              //  Tipo de atualizacao de parcela                  173 174     9(02)
            $conteudo .= self::complementoRegistro(10, "brancos");                          //  Caracteres extras da mensagem                   175 183     9(10)
        }
        $conteudo .= self::limit($emp['idRemessa'], 8, "zeros");                        //  Numero de remessa (sequencial)                      184 191     9(08)
        $conteudo .= date("dmY");                                                       //  Data de gravacao da remessa                         192 199     9(08)
        $conteudo .= self::complementoRegistro(8, "zeros");                             //  Data do credito (apenas para retorno)               200 207     9(08)
        $conteudo .= self::complementoRegistro(33, "brancos");                          //  Uso exclusivo do CNAB                               208 240     9(33)
        $conteudo .= chr(13) . chr(10);                                                 //  quebra de linha

        $vl_total = 0.00;
        $sequencial = 1;
        foreach ($parcelas as $k => $parcela) {
            $vl_parcela = str_pad(str_replace(array(".", ","), "", number_format($parcela["vl_parcela"], 2)), 15, '0', STR_PAD_LEFT);
            //Condicao criaca para que essa alteracao do tipo de juros/multa sendo porcentagem/fixo seja apenas para CEF por enquanto
            $registro = 1;
            if ($emp["tp_juros"] != 'S') {
                $juros = ($emp["tp_juros"] == 'P') ? (float)$emp["juros"] * (float)$parcela["vl_parcela"] / 100 : $emp["juros"];
            } else {
                $juros = 0;
            }
            $juros = str_replace(array(".", ","), "", number_format($juros, 2));
            $multa = str_replace(array(".", ","), "", number_format($emp["multa"], 2));
            $vl_total += $vl_parcela;
            if ($emp['cd_banco'] == '237' || $emp['cd_banco'] == '741')
                if ($tipoRegistro == '1')
                    $cpfCnpj = substr($cpfCnpj, 0, 9) . '0000' . substr($cpfCnpj, 9, 3);
            $nossoNumero = self::geraNossoNumero($emp['cd_banco'], $parcela, $emp);
            $dias_permanencia = (isset($emp['qt_dias_permanencia']) && (int)$emp['qt_dias_permanencia'] > 5) ? $emp['qt_dias_permanencia'] : 90;

            $contrato['tp_parcela'] = $parcela['tp_parcela'];
            $contrato['dt_parcela'] = $parcela['dt_parcela'];

            $descontoParcela = Parcela::getDescontoParcela($contrato, $parcela['vl_parcela'] );
            
            if((float)$parcela['vl_parcela_2avia'] > 0 && !$parcela['desconto_2via'])
                $descontoParcela['vlDesconto'] = 0;
            $vlDesconto = str_pad(str_replace(array(".", ","), "", number_format($descontoParcela['vlDesconto'], 2)), 15, '0', STR_PAD_LEFT);
            

            ## SEGMENTO P (OBRIGATORIO)                                                 ##  NOME DO CAMPO                                       POSICAO    TAMANHO
            $conteudo .= $emp['cd_banco'];                                              //  Codigo do banco                                     001 003     9(03)
            $conteudo .= self::sequencial($nrArquivo, 4);                               //  Lote do servico                                     004 007     9(04)
            $conteudo .= '3';                                                           //  Registro detalhe                                    008 008     9(01)
            $conteudo .= self::sequencial($sequencial, 5);                              //  Numero sequencial do registro no lote               009 013     9(05)
            $conteudo .= 'P';                                                           //  Codigo do segmento                                  014 014     X(01)
            $conteudo .= ' ';                                                           //  Uso do banco                                        015 015     X(01)
            $conteudo .= $codMovimento;                                                 //  Codigo de movimentacao (Informado ao chamar funcao) 016 017     9(02)
            
            $bancoRibeiraoPreto = '';
            if ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO)
                $bancoRibeiraoPreto .= $emp["agencia"];                                        //  Agencia                                             018 022     9(05)
            else 
                $conteudo .= self::limit($emp["agencia"], 5, "zeros");                      //  Agencia                                             018 022     9(05)
            
            if ($emp['cd_banco'] == '341')  //Banco Itau
                $conteudo .= self::complementoRegistro(1, "brancos");                   // Complemento de registro                          023 023        X(01)
            elseif ($emp['cd_banco'] == '748')  //Banco Sicredi
                $conteudo .= ($emp["agencia_dv"])?$emp["agencia_dv"]:' ';               //  Dv da agencia                                       023 023     9(01)
            elseif ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO)
                $bancoRibeiraoPreto .= $emp["agencia_dv"];                                  //  Dv da agencia                                       023 023     9(01)
            else
                $conteudo .= self::limit($emp["agencia_dv"], 1, "zeros");               //  Dv da agencia                                       023 023     9(01)
            
            if ($emp['cd_banco'] == '104') { //Codicional se e caixa economica incluindo registro especifico
                
                if (strlen($emp['cd_cedente']) < 7) {
                    $conteudo .= self::limit($emp['cd_cedente'], 6, "zeros");               //10.3P Codigo do convenio do cedente                       024 029     9(06)
                    $conteudo .= self::complementoRegistro(8, "zeros");                     //11.3P Uso exclusivo da caixa                              030 037     9(08)
                } else {
                    $conteudo .= self::limit($emp['cd_cedente'], 7, "zeros");               //10.3P Codigo do convenio do cedente                       024 029     9(06)
                    $conteudo .= self::complementoRegistro(7, "zeros");                     //11.3P Uso exclusivo da caixa                              030 037     9(08)
                }
            } else {

                if ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO)
                    $bancoRibeiraoPreto .= $emp['cd_cedente'];                                  //  Conta corrente                                      024 035     9(12)
                else
                    $conteudo .= self::limit($emp['conta_corrente'], 12, "zeros");          //  Conta corrente                                      024 035     9(12)
                
                if ($emp['cd_banco'] == '341')  //Banco Itau
                    $conteudo .= self::complementoRegistro(1, "brancos");               // Complemento de registro                          036 036        X(01)
                elseif ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO)
                    $bancoRibeiraoPreto .= '';
                else
                    $conteudo .= self::limit($emp['conta_corrente_dv'], 1, "zeros");    //  Digito verificador da Conta corrente                036 036     9(01)
                
                if ($emp['cd_banco'] == '237' || $emp['cd_banco'] == '756')
                    $conteudo .= self::complementoRegistro(1, "brancos");               //  Digito do banco (padrao ' ')                        037 037     9(01)
                elseif ($emp['cd_banco'] == '341')
                    $conteudo .= self::limit($emp['conta_corrente_dv'], 1, "brancos");  // Digito do banco (padrao brancos itau)          037 037        X(01)
                elseif ($emp['cd_banco'] == Banco::BANCO_BANPARA)                                       //  Digito do banco                        037 037     9(01)
                    $conteudo .= self::limit($emp["agencia_dv"], 1, "zeros");
                elseif ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO)
                    $bancoRibeiraoPreto .= '';
                else
                    $conteudo .= self::complementoRegistro(1, "zeros");                 //  Digito do banco (padrao 0)                          037 037     9(01)
            }

            if ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO)
                $conteudo .= self::limit('050' . $bancoRibeiraoPreto, 20, "brancos");

            if ($emp['cd_banco'] == '341') {
                $conteudo .= self::limit($nossoNumero, 20, "zeros");                    //  Identific. do titulo no banco                       038 057     9(20)
                $conteudo .= self::complementoRegistro(5, 'zeros');                     //  Complemento de registro                             058 062     9(01)
                
                if ($parcela['nr_boleto']) {
                    $conteudo .= self::limit($parcela["nr_boleto"], 10, "zeros"); //  Numero do documento de cobranca                     063 072     9(10)
                } else {
                    $conteudo .= self::limit($parcela["id_parcela_contrato"], 10, "zeros"); //  Numero do documento de cobranca                     063 072     9(10)
                }

                $conteudo .= self::complementoRegistro(5, "brancos");                   //  Reservado para uso do banco                         073 077     X(05)
            } else {
                if ($emp['cd_banco'] == Banco::BANCO_SAFRA) {                                    //  Identific. do titulo no banco                       038 057     9(20)
                    $conteudo .= substr($nossoNumero, -9); //  Identific. do titulo no banco                       038 057     9(20)
                    $conteudo .= self::complementoRegistro(11, "brancos");
                } elseif ($emp['cd_banco'] == Banco::BANCO_BANPARA){
                    $conteudo .= self::limit($nossoNumero, 20, "zeros");                        //  Identific. do titulo no banco                       038 057     9(20)
                } elseif ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO) {
                    $conteudo .= '5';                                                           // Servio de Classificao (posio 38): informar ?5? (SOMENTE BANCO RIBEIRAO)
                    $conteudo .= self::complementoRegistro(5, 'zeros');                        //  Modalidade com correspondente: informar ?000? ;      39 a 43     9(19)
                    $conteudo .= '110';                                                        //Modalidade com banco cedente informar ?112?;           44 a 46
                    $conteudo .= self::complementoRegistro(5, 'zeros');                         // Identificao do ttulo no banco preencher com zeros  47 a 57
                    
                    $dvRibeiraoPreto = self::digitoVerificadorRibeiraoPreto($emp['agencia'], $emp['carteira'], $parcela['nr_boleto']);

                    $conteudo .= self::limit($parcela['nr_boleto'].$dvRibeiraoPreto, 11, "zeros");                        //  Identific. do titulo no banco                       047 057     9(19)
                } else {                                                                        //  Identific. do titulo no banco                       038 057     9(20)
                    $conteudo .= self::limit($nossoNumero, 20, "zeros");                        //  Identific. do titulo no banco                       038 057     9(20)
                }
                
                $conteudo .= $emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO ? '1' : $codCarteira;  //  Cd da carteira 1= cob. simples. 2= cob. caucionada  058 058     9(01)

                if ($emp['cd_banco'] == '756') { //Sicoob
                    $conteudo .= '0';                                                   //  Forma de cadastramento do titulo no banco           059 059     9(01)
                    $conteudo .= ' ';                                                   //  Tipo de documento                                   060 060     9(01)
                }  elseif ($emp['cd_banco'] == Banco::BANCO_BANPARA){
                    $conteudo .= '1';                                                       //  Tipo de documento  059 059     9(01)
                    $conteudo .= '1';                                                       //  Tipo de documento  060 060     9(01)
                } elseif ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO) {
                    $conteudo .= '1';                                                       //  Tipo de documento  059 059     9(01)
                    $conteudo .= '1';                                                       //  Tipo de documento  060 060     9(01)
                } else {
                    $conteudo .= $registro;                                             //  Forma de cadastramento do titulo no banco           059 059     9(01)
                    $conteudo .= '2';                                                   //  Tipo de documento                                   060 060     9(01)
                }
                
                $conteudo .= $emissor;   //  Emissao do bloqueto 2                061 061     9(01)                          
                
                if ($emp['cd_banco'] == '104')
                    $conteudo .= '0';   //  Identificacao da distribuicao 1 = Banco distribui   062 062     9(01)
                else
                    $conteudo .= $emissor;   //  Identificacao da distribuicao 1 = Banco distribui   062 062     9(01)
                
                if ($emp['cd_banco'] == '756' || $emp['cd_banco'] == '004') { //Sicoob ou Banco Nordeste
                    if ($parcela['nr_boleto']) {
                        $conteudo .= self::limit($parcela["nr_boleto"], 15, "zeros");     //  Numero do documento de cobranca                     063 077     9(11)
                    } else {
                        $conteudo .= self::limit($parcela["id_parcela_contrato"], 15, "zeros");     //  Numero do documento de cobranca                     063 077     9(11)
                    }
                } else {
                    if ($emp['cd_banco'] == '748' || $emp['cd_banco'] == '741') {
                        $conteudo .= substr($parcela["nr_boleto"], -6);            //  Numero do documento de cobranca                     063 073     9(11)
                        $conteudo .= self::complementoRegistro(5, "zeros");     //  Numero do documento de cobranca                     063 073     9(11)
                    } elseif ($emp['cd_banco'] == Banco::BANCO_SAFRA){
                        if ($parcela['nr_boleto']) {
                            $conteudo .= self::limit($parcela["nr_boleto"], 10, "zeros");
                        } else {
                            $conteudo .= self::limit($parcela["id_parcela_contrato"], 10, "zeros");
                        }
                        $conteudo .= self::complementoRegistro(1, "brancos");
                    } elseif ($emp['cd_banco'] == '099') {
                        $conteudo .= self::limit($parcela["id_parcela_contrato"], 15, "zeros");
                    }else {
                        if ($parcela['nr_boleto']) {
                            $conteudo .= self::limit($parcela["nr_boleto"], 11, "zeros");     //  Numero do documento de cobranca                     063 073     9(11)
                        } else {
                            $conteudo .= self::limit($parcela["id_parcela_contrato"], 11, "zeros");     //  Numero do documento de cobranca                     063 073     9(11)
                        }
                    }
                    if ($emp['cd_banco'] == '099') {
                        $conteudo .= self::complementoRegistro(0, "brancos");
                    } else {
                        $conteudo .= self::complementoRegistro(4, "brancos");                       //  Reservado para uso do banco                         074 077     9(04)
                    }
                }                  //  Reservado para uso do banco                         074 077     9(04)
            }
            $conteudo .= date("dmY", strtotime($parcela["dt_parcela"]));                //  Vencimento da prestacao                             078 085     9(08)
            $conteudo .= $vl_parcela;                                                   //  Valor da prestacao                                  086 100     9(15)
            $conteudo .= self::complementoRegistro(5, "zeros");                         //  Agencia responsavel pela cobranca (USO DA CAIXA)    101 105     9(05)
            if ($emp['cd_banco'] == '756') { //Condicional do Sicoo
                $conteudo .= ' ';                                                       //  Digito verificador da agencia (USO DA CAIXA)        106 106     9(01)
            } else {
                $conteudo .= self::complementoRegistro(1, "zeros");                     //  Digito verificador da agencia (USO DA CAIXA)        106 106     9(01)
            }
            if ($emp['cd_banco'] == '001' || $emp['cd_banco'] == '756' || $emp['cd_banco'] == '004' || $emp['cd_banco'] == Banco::BANCO_BRADESCO || $emp['cd_banco'] == Banco::BANCO_CAIXA_ECONOMICA || $emp['cd_banco'] == Banco::BANCO_BANPARA || $emp['cd_banco'] == Banco::BANCO_SAFRA || $emp['cd_banco'] == Banco::BANCO_UNIPRIME) {  //  Especie do titulo (99 = outros)                     107 108     9(02)
                $conteudo .= '02';                                                      //  Especie do titulo (99 = outros)                     107 108     9(02)
            } elseif ($emp['cd_banco'] == '748' && $emp['especie_documento'] == 'DMI') {
                $conteudo .= '03';                                                      //  SOMENTE PARA O BANCO SICREDI. Especie do titulo (03 = DMI - Duplicata mercantil por indicao)                     107 108     9(02) 
            } else {
                $conteudo .= '99';                                                      //  Especie do titulo (99 = outros)                     107 108     9(02)
            }
            $conteudo .= "N";                                                           //  Aceite A = Aceite N = Nao aceite                    109 109     X(01)
            $conteudo .= date("dmY");                                                   //  Data da emissao da parcela                          110 117     9(08)
            $conteudo .= "1";                                                           //  Cidigo do juros/mora (1 = diario)                   118 118     9(01)
            if (in_array($emp['cd_banco'], ['104', '422'])) { //Sicoob e Caixa
                $conteudo .= date("dmY", strtotime($parcela["dt_parcela"]));            //  Data do juros. branco = data vencimento             119 126     9(08)
            } elseif ($emp['cd_banco'] == '756') {
                $conteudo .= date("dmY", strtotime($parcela["dt_parcela"] . ' + 1 days'));  //  Data do juros, do Sicoob + 1 dia                119 126
            } else {
                $conteudo .= self::complementoRegistro(8, "zeros");                     //  Data do juros. branco = data vencimento             119 126     9(08)
            }
            $conteudo .= self::limit($juros, 15, "zeros");                              //  Juros por dia                                       127 141     9(13)
            if ($descontoParcela['flDesconto'] != 'S' && (float)$descontoParcela['vlDesconto'] > 0) {
                $conteudo .= "1";                                                       //  Codigo do desconto. 1 = valor 2 = percentual        142 142     9(01)
                $conteudo .= date("dmY", strtotime($parcela["dt_parcela"])); 
            } else {
                $conteudo .= "0";                                                       //  Codigo do desconto. 1 = valor 2 = percentual        142 142     9(01)
                $conteudo .= self::complementoRegistro(8, "zeros");                     //  Data do desconto                                    143 150     9(08)
            }
            $conteudo .= $vlDesconto;                                                   //  Valor do desconto                                   151 165     9(13)
            $conteudo .= self::complementoRegistro(15, "zeros");                        //  Valor IOF a pagar                                   166 180     9(13)
            $conteudo .= self::complementoRegistro(15, "zeros");                        //  Valor do abatimento                                 181 195     9(13)
            if ($emp['cd_banco'] == '748' || $emp['cd_banco'] == '741') {
                $conteudo .= self::limit($parcela["nr_boleto"], 25, "zeros");     //  Codigo de identificacao da parcela na empresa       196 220     9(25)
            } else {
                if ($parcela['nr_boleto']) {
                    $conteudo .= self::limit($parcela["nr_boleto"], 25, "zeros");     //  Codigo de identificacao da parcela na empresa       196 220     9(25)
                } else {
                    $conteudo .= self::limit($parcela["id_parcela_contrato"], 25, "zeros");     //  Codigo de identificacao da parcela na empresa       196 220     9(25)
                }
            }
            $conteudo .= "3";                                                           //  Codigo para protesto (3 = sem protesto)             221 221     9(01)
            $conteudo .= self::complementoRegistro(2, "zeros");                         //  Numero de dias para protesto                        222 223     9(02)
            
            if ($emp['cd_banco'] == '756') //Condicional do Sicoob
                $conteudo .= "0";                                                       //  Codigo de Baixa do boleto (2 = nao baixar)          224 224     9(01)
            else
                $conteudo .= "1";                                                       //  Codigo de Baixa do boleto (2 = nao baixar)          224 224     9(01)
            
            if ($emp['cd_banco'] == '341') { //ITAU
                $conteudo .= self::limit($dias_permanencia, 2, "zeros");                //  Numero de dias para baixa                           225 226     9(02)
                $conteudo .= '0';                                                       //  Numero de dias para baixa                           227 227     9(01)
            } elseif ($emp['cd_banco'] == '756') { //SICOOB
                $conteudo .= '   ';                                                     //  Numero de dias para baixa                           225 227     X(03)
            } else {
                $conteudo .= self::limit($dias_permanencia, 3, "zeros");                //  Numero de dias para baixa                           225 227     X(03)
            }
            $conteudo .= "09";                                                          //  Codigo da moeda (09 = real)                         228 229     9(02)
            $conteudo .= self::complementoRegistro(10, "zeros");                        //  Numero identificador do contrato na empresa         230 239     9(10)
            
            if ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO)
                $conteudo .= "1";                           //  Uso exclusivo do FEBRABAN                           240 240     9(01)
            else
                $conteudo .= self::complementoRegistro(1, "brancos");                       //  Uso exclusivo do FEBRABAN                           240 240     9(01)
            
            $conteudo .= chr(13) . chr(10);                                             //  essa e a quebra de linha

            $sequencial++;

            $cep = self::retirar_caracteres_especiais($parcela['nr_cep']);
            $nrCep = substr($cep, 0, 5);
            $sufixoCep = substr($cep, 5, 3);

            ## SEGMENTO Q (OBRIGATORIO)                                                 ##  NOME DO CAMPO                                       POSICAO    TAMANHO
            $conteudo .= $emp['cd_banco'];                                              //  Codigo do banco                                     001 003     9(03)
            $conteudo .= self::sequencial($nrArquivo, 4);                               //  Lote do servico                                     004 007     9(04)
            $conteudo .= '3';                                                           //  Registro detalhe                                    008 008     9(01)
            $conteudo .= self::sequencial($sequencial, 5);                              //  Numero sequencial do registro no lote               009 013     9(05)
            $conteudo .= 'Q';                                                           //  Codigo do segmento                                  014 014     X(01)
            $conteudo .= ' ';                                                           //  Uso do banco                                        015 015     X(01)
            $conteudo .= $codMovimento;                                                 //  Codigo de movimentacao (Informado ao chamar funcao) 016 017     9(02)
            $conteudo .= $tipoRegistro;                                                 //  Tipo de inscricao do Sacado                         018 018     9(01)
            $conteudo .= self::limit($cpfCnpj, 15, "zeros");                            //  Numero de inscricao do Sacado                       019 033     9(15)
            $conteudo .= self::limit(Helper::removeCaracteresEspeciais($parcela['nm_pessoa']), 40, 'brancos');             //  Nome do Sacado                                      034 073     X(40)
            $conteudo .= self::limit(Helper::removeCaracteresEspeciais($parcela['endereco']), 40, 'brancos');              //  Endereco do Sacado                                  074 113     X(40)
            $conteudo .= self::limit(Helper::removeCaracteresEspeciais($parcela['nm_bairro']), 15, "brancos");             //  Bairro do Sacado                                    114 128     X(15)
            $conteudo .= self::limit($nrCep, 5, ($emp['cd_banco'] == '341') ? "zeros" : "brancos");  //  CEP do Sacado                                       129 133     9(05)
            $conteudo .= self::limit($sufixoCep, 3, ($emp['cd_banco'] == '341') ? "zeros" : "brancos");//  Sufixo do Cep do sacado                             134 136     9(03)
            $conteudo .= self::limit(Helper::removeCaracteresEspeciais($parcela['cidade']), 15, "brancos");                //  Nome da Cidade do Sacado                            137 151     X(15)
            $conteudo .= self::limit($parcela['uf'], 2, "brancos");                     //  Uf do Sacado                                        152 153     X(02)
            
            $tipoPagador = '0';                                                             //  Tipo de inscricao do pagador (se diferente)         154 154     9(01)

            if ($emp['cd_banco'] == Banco::BANCO_SICREDI)
                $tipoPagador = $tipoRegistro;                                              //  Tipo de inscricao do pagador (1=Pessoa Fsica; 2=Pessoa Jurdica), no momento essa condio s funciona para o SICREDI         154 154     9(01)

            $conteudo .= $tipoPagador;

            if ($emp['cd_banco'] == Banco::BANCO_SICREDI) {
                $conteudo .= self::complementoRegistro(15, "brancos");                      //  Numero do documento do pagador (se diferente), SOMENTE PARA O SICREDI       155 169     9(15)
            } else {
                $conteudo .= self::complementoRegistro(15, "zeros");                        //  Numero do documento do pagador (se diferente)       155 169     9(15)
            }
            $conteudo .= self::complementoRegistro(40, "brancos");                      //  Nome do pagador (se diferente)                      170 209     X(40)
            if ($emp['cd_banco'] == '104'){                                                       //  Codigo do Banco Correspondente na Compensacao       210 212     9(03)
                $conteudo .= self::complementoRegistro(3, "brancos");                   // SE FOR CEF COLOCAR BRANCOS
            }else{
                $conteudo .= self::complementoRegistro(3, "zeros");                     // DEMAIS BANCOS, COLOCA ZERO!
            }
            $conteudo .= self::complementoRegistro(20, "brancos");                      //  Nosso Nr no Banco Correspondente                    213 232     X(20)
            $conteudo .= self::complementoRegistro(8, "brancos");                       //  Reservado para uso do banco                         233 240     X(08)
            $conteudo .= chr(13) . chr(10);                                             //  essa e a quebra de linha

            $sequencial++;

            ## SEGMENTO R (OPCIONAL)                                                    ##  NOME DO CAMPO                                       POSICAO    TAMANHO
            $conteudo .= $emp['cd_banco'];                                              //  Codigo do banco                                     001 003     9(03)
            $conteudo .= self::sequencial($nrArquivo, 4);                               //  Lote do servico                                     004 007     9(04)
            $conteudo .= '3';                                                           //  Registro detalhe                                    008 008     9(01)
            $conteudo .= self::sequencial($sequencial, 5);                              //  Numero sequencial do registro no lote               009 013     9(05)
            $conteudo .= 'R';                                                           //  Codigo do segmento                                  014 014     X(01)
            $conteudo .= ' ';                                                           //  Uso do banco                                        015 015     X(01)
            $conteudo .= $codMovimento;                                                 //  Codigo de movimentacao (Informado ao chamar funcao) 016 017     9(02)
            $conteudo .= '0';                                                           //  Tipo de Desconto2 do titulo                         018 018     9(01)
            $conteudo .= self::complementoRegistro(8, "zeros");                         //  Data do desconto2 do titulo                         019 026     9(08)
            $conteudo .= self::complementoRegistro(15, "zeros");                        //  Valor do desconto2 do titulo                        027 041     9(13)
            $conteudo .= '0';                                                           //  Tipo de Desconto3 do titulo                         042 042     9(01)
            $conteudo .= self::complementoRegistro(8, "zeros");                         //  Data do desconto3 do titulo                         043 050     9(08)
            $conteudo .= self::complementoRegistro(15, "zeros");                        //  Valor do desconto3 do titulo                        051 065     9(15)
            if ($emp['cd_banco'] == '748') { //Sicredi aceita apenas percentual
                $conteudo .= '2';                                                       //  Tipo de Multa (2 Percentual)                        066 066     9(01)
            } else {
                if ($emp["tp_multa"] == 'P') {
                    $conteudo .= '2';                                                   //  Tipo de Multa (0 sem multa/1 vlr fixo/2 Perc)       066 066     9(01)
                } elseif ($emp["tp_multa"] == 'V') {
                    $conteudo .= '1';                                                   //  Tipo de Multa (0 sem multa/1 vlr fixo/2 Perc)       066 066     9(01)
                } else {
                    $conteudo .= '0';                                                   //  Tipo de Multa (0 sem multa/1 vlr fixo/2 Perc)       066 066     9(01)
                }
            }
            if (in_array($emp['cd_banco'], ['104', '341','001','237', '741', '422'])) {
                $conteudo .= self::limit(date("dmY", strtotime($parcela["dt_parcela"])), 8, "zeros"); //Data da multa do titulo                 067 074     9(08)
            } elseif ($emp['cd_banco'] == '756') {
                $conteudo .= date("dmY", strtotime($parcela["dt_parcela"] . ' + 1 days'));  //  Data do multa, do Sicoob + 1 dia                067 074
            } else {
                $conteudo .= self::complementoRegistro(8, "zeros");                     //Data da multa do titulo                               067 074     9(08)
            }
            $conteudo .= self::limit($multa, 15, "zeros");                              //  Valor da multa do titulo (porcento)                 075 089     9(15)
            $conteudo .= self::complementoRegistro(10, "brancos");                      //  Informacoes do Pagador                              090 099     X(10)
            $conteudo .= self::complementoRegistro(40, "brancos");                      //  Mensagem nr 3                                       100 139     X(40)
            $conteudo .= self::complementoRegistro(40, "brancos");                      //  Mensagem nr 4                                       140 179     X(40)
            if (in_array($emp['cd_banco'], ['104', '341', '756'])) { //Condicional do Bradesco, Itau e Sicoob
                $conteudo .= self::complementoRegistro(20, "brancos");                  //  Uso Exclusivo FEBRABAN/CNAB                         180 199     X(20)
                $conteudo .= self::complementoRegistro(8, "zeros");                     //  Cod. Ocor. do Pagador                               200 207     9(08)
                $conteudo .= self::complementoRegistro(3, "zeros");                     //  Cod. do Banco na Conta do Debito                    208 210     9(03)
                $conteudo .= self::complementoRegistro(5, "zeros");                     //  Codigo da Agencia do Debito                         211 215     9(05)
                $conteudo .= self::complementoRegistro(1, "brancos");                   //  Digito Verificador da Agancia                       216 216     X(01)
                $conteudo .= self::complementoRegistro(12, "zeros");                    //  Conta Corrente para Debito                          217 228     9(12)
                $conteudo .= self::complementoRegistro(1, "brancos");                   //  Digito Verificador da Conta                         229 229     X(01)
                $conteudo .= self::complementoRegistro(1, "brancos");                   //  Digito Verificador Ag/Conta                         230 230     X(01)
                if ($emp['cd_banco'] == '237') { //Condicional do bradesco
                    $conteudo .= '2';                                                   //  Aviso para Dibito Automatico                        231 231     9(01)
                } elseif ($emp['cd_banco'] == '741') {
                    $conteudo .= '0';
                } else {
                    $conteudo .= self::complementoRegistro(1, "zeros");                 //  Aviso para Dibito Automatico                        231 231     9(01)
                }
                $conteudo .= self::complementoRegistro(9, "brancos");                   //  Uso Exclusivo FEBRABAN/CNAB                         232 240     X(09)
            } else {
                $conteudo .= self::complementoRegistro(50, "brancos");                  //  E-mail do pagador                                   180 229     X(50)
                $conteudo .= self::complementoRegistro(11, "brancos");                  //  Reservado para uso do banco                         231 240     X(11)
            }
            $conteudo .= chr(13) . chr(10);                                             //  quebra de linha

            $sequencial++;

            if ($emp['cd_banco'] == '748' && $emp['chave_pix']) {
                ##SEGMENTO Y (PARA REMESSA HBRIDA)
                $conteudo .= $emp['cd_banco'];
                $conteudo .= self::sequencial($nrArquivo, 4);
                $conteudo .= '3';
                $conteudo .= self::sequencial($sequencial, 5);
                $conteudo .= 'Y';
                $conteudo .= ' ';
                $conteudo .= $codMovimento;
                $conteudo .= '04';
                $conteudo .= self::complementoRegistro(62, "brancos");
                $conteudo .= $emp['chave_pix'];                                             // chave pix da agncia
                $conteudo .= self::complementoRegistro(35, "brancos");
                $conteudo .= self::complementoRegistro(88, "brancos");
                $conteudo .= chr(13) . chr(10);                                             //  essa e a quebra de linha

                $sequencial++;
            }

        }
        $vl_total = number_format($vl_total, 2);
        $vl_total = str_replace(array(".", ","), "", $vl_total);

        ##  TRAILER DO LOTE (OBRIGATORIO) - REGISTRO 5
        #  NOME DO CAMPO                                                                #SIGNIFICADO                                            #POSICAO    #TAMANHO
        $conteudo .= $emp['cd_banco'];                                                  //  Codigo banco                                        001 003     9(03)
        $conteudo .= self::sequencial($nrArquivo, 4);                                   //  Lote                   Lote de servico            004 007     9(04)
        $conteudo .= '5';                                                               //  Registro trailer do lote                            008 008     9(01)
        $conteudo .= self::complementoRegistro(9, "brancos");                           //  Filler                                              009 017     X(09)
        
        if ($emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO)
            $conteudo .= self::complementoRegistro(6, "zeros");                          //  Numero de registros no lote                        018 023     9(06)
        else
            $conteudo .= self::limit($sequencial + 1, 6, "zeros");                          //  Numero de registros no lote                        018 023     9(06)
        
        if ($emp['cd_banco'] == '237' || $emp['cd_banco'] == '741' || $emp['cd_banco'] == '748') { //Condicional do bradesco, banco ribeiro preto e sicredi
            $conteudo .= self::complementoRegistro(6, "zeros");                         //  Numero de titulos no lote                          024 029     9(06)
        } else {
            $conteudo .= self::limit(count($parcelas), 6, "zeros");                     //  Numero de titulos no lote                          024 029     9(06)
        }
        if ($emp['cd_banco'] == '748' || $emp['cd_banco'] == '741') { //SICREDI e BANCO RIBEIRO PRETO
            $conteudo .= self::complementoRegistro(17, "zeros");                               //  Valor total dos titulos no lote                     030 046     9(17)
        } else {
            $conteudo .= self::limit($vl_total, 17, "zeros");                               //  Valor total dos titulos no lote                     030 046     9(17)
        }
        $conteudo .= self::complementoRegistro(6, "zeros");                             //  Numero de titulos caucionados no lote              047 052     9(06)
        $conteudo .= self::complementoRegistro(17, "zeros");                            //  Valor total dos titulos caucionados no lote         053 069     9(17)
        $conteudo .= self::complementoRegistro(6, "zeros");                             //  Numero de titulos descontados no lote              070 075     9(06)
        $conteudo .= self::complementoRegistro(17, "zeros");                            //  Valor total dos titulos descontados no lote         076 092     9(17)
        if ($emp['cd_banco'] == '341' || $emp['cd_banco'] == '756' || $emp['cd_banco'] == Banco::BANCO_SAFRA || $emp['cd_banco'] == Banco::BANCO_RIBEIRAO_PRETO) { //Condicional do Itau // ou Banco Sicob
            $conteudo .= self::complementoRegistro(6, "zeros");                         //  Uso exclusivo do banco                              093 098     9(06)
            $conteudo .= self::complementoRegistro(17, "zeros");                        //  Uso exclusivo do banco                              099 115     9(17)
            $conteudo .= self::complementoRegistro(8, "brancos");                       //  Uso exclusivo do banco                              116 123     9(08)
        }  else {
            $conteudo .= self::complementoRegistro(31, "brancos");                      //  Uso exclusivo do CNAB                               093 123     9(31)
        }
        $conteudo .= self::complementoRegistro(117, "brancos");                         //  Uso exclusivo do CNAB                               124 240     9(117)
        $conteudo .= chr(13) . chr(10);                                                 //  essa e a quebra de linha
        return $conteudo;
    }
}