<?php

class LoginOutOfTimeException extends Exception
{
}

class LoginNotMatchException extends Exception
{
}

class LoginDisabledUserException extends Exception
{
}

class LoginSemPermissaoException extends Exception
{
}

class Login
{
    private $usuario;
    private $login;
    private $senha;

    const SESSION_NAME = "usuario";

    /**
     * Realiza a autenticao do usurio na tela de login do sistema
     * Verifica se o usurio existe, se est ativo e se ele est vlido
     * @throws LoginDisabledUserException
     * @throws LoginNotMatchException
     * @throws LoginOutOfTimeException
     */
    public function autenticar()
    {
        $this->usuario = Db_Usuario::fetchByLogin($this->login, $this->senha);

        if ($this->usuario) {
            if (!$this->checarUsuarioAtivo() && $this->checarDataLimite()) {
                if ($this->checarLimiteHorario() && $this->checarLimiteHorario()) {

                    $aclUsuario = (new ACL())->verificaPerfilAcesso($this->usuario->getPerfilId());

                    if ($aclUsuario) {
                        $_SESSION[self::SESSION_NAME] = $this->usuario;
                    } else {
                        throw new LoginSemPermissaoException();
                    }
                } else {
                    throw new LoginOutOfTimeException();
                }
            } else {
                throw new LoginDisabledUserException();
            }
        } else {
            throw new LoginNotMatchException();
        }
    }

    /**
     * Verifica se o usurio est ativo ou no
     * @return bool
     */
    public function checarUsuarioAtivo()
    {
        $user = $this->usuario;
        return ($user->getStatus() == 0);
    }


    /**
     * Verifica se o usurio expirou ou se ainda  vlido
     * @return bool
     */
    public function checarDataLimite()
    {
        $user = $this->usuario;
        return ($user->getDataExpiracao() && $user->getDataExpiracao() >= date("Y-m-d H:i:s"));
    }

    /**
     * Renora se o usurio est em um horrio vlido
     * @return boolean
     */
    public function checarLimiteHorario()
    {
        $user = $this->usuario;
        $hora = date("H:i:s");
        if ($user->getHoraInicio() && $user->getHoraTermino()) {
            if ($hora >= $user->getHoraInicio() && $hora <= $user->getHoraTermino()) {
                return true;
            } else {
                return false;
            }
        } else {
            return true;
        }
    }

    public static function isLogado()
    {
        $usuario = self::getUsuario();
        if ($usuario && $usuario instanceof Db_Usuario)
            return true;
        else return false;
    }

    //checa se o usurio logado tem o status de super user
    public static function isSuper()
    {
        $usuario = self::getUsuario();
        if ($usuario->getSuper() == '1') {
            return true;
        } else {
            return false;
        }
    }

    public static function getUsuario()
    {
        return $_SESSION[self::SESSION_NAME];
    }

    public function getLogin()
    {
        return $this->login;
    }

    public function getSenha()
    {
        return $this->senha;
    }

    public function setLogin($login)
    {
        $this->login = $login;
    }

    public function setSenha($senha)
    {
        $this->senha = $senha;
    }

    public static function logout()
    {
        unset($_SESSION[self::SESSION_NAME]);
        unset($_SESSION['avisos']);
        return;
    }
}