<?php

class ContratoException extends Exception
{

}

class Contrato extends Zend_Db_Table_Abstract
{
    const CONTRATO_GERADO = '1';
    const CONTRATO_NAO_GERADO = '0';
    const CONTRATO_TODOS = '3';

    static $contratoInstacia;
    protected $_name = TB_CONTRATO;

    /**
     * Busca os contratos pelo id do empreendimento
     *
     * @param $empreendimento = id do empreendimento
     * @param string $tipo = tipo de contrato (gerados, nao gerados ou todos)
     * @param null $distratado = true para contratos distratos ou false para no distratados
     * @param null $order = ordenao da lista
     * @param bool $aprovados = true para contratos aprovados ou false para no aprovados
     * @return array|Zend_Db_Select = lista dos contratos
     */
    public function getContrato($id)
    {
        return self::getInstance()->getDefaultAdapter()->select()
            ->from(array('c' => TB_CONTRATO), '*')
            ->where('c.id = ?', $id)
            ->query()->fetch();
    }


    /**
     * Busca os contratos pelo id do empreendimento
     *
     * @param $empreendimento = id do empreendimento
     * @param string $tipo = tipo de contrato (gerados, nao gerados ou todos)
     * @param null $distratado = true para contratos distratos ou false para no distratados
     * @param null $order = ordenao da lista
     * @param bool $aprovados = true para contratos aprovados ou false para no aprovados
     * @return array|Zend_Db_Select = lista dos contratos
     */
    public static function byEmpreendimento($empreendimento, $tipo = self::CONTRATO_TODOS, $distratado, $order = null, $aprovados = false)
    {
        $contrato = self::getInstance();
        $contratos = $contrato->getDefaultAdapter()->select()->from(array('c' => TB_CONTRATO), array('c.id as id', 'c.id_pessoa', 'c.fl_distrato', 'id_banco'))
            ->join(array('l' => TB_LOTES), 'c.id_lote = l.id', array('l.id as id_lote', 'quadra', 'lote'))
            ->join(array('e' => TB_EMPREENDIMENTO), 'l.id_empreendimento = e.id', array('e.id as id_empreendimento'))
            ->join(array('p' => TB_PESSOA), 'p.id = c.id_pessoa', array('p.nm_pessoa'))
            ->where('e.id = ?', $empreendimento);
        if ($order)
            $contratos = $contratos->order($order);
        //testa o parametro do tipo de contrato (gerado parcelas ou no)
        if ($tipo != self::CONTRATO_TODOS && $tipo != null)
            $contratos = $contratos->where('c.fl_gerado = ?', $tipo);
        //testa se vai procurar os distratados ou no (default retorna todos)
        if ($distratado != 2)
            $contratos = $contratos->where("c.fl_distrato = '$distratado'");
        if ($aprovados)
            $contratos = $contratos->where("c.fl_aprovar_contrato = '1'");
        $contratos = $contratos->query()->fetchAll();
        return $contratos;

    }

    /**
     * Busca os contratos do empreendimento e com o banco setado
     * @param $empreendimento , $banco
     * @param string $tipo
     * @param null $distratado
     * @param null $order
     * @param bool $aprovados
     * @return array|Zend_Db_Select
     */
    public static function byBancoEmpreendimento($empreendimento, $banco = null, $tipo = self::CONTRATO_TODOS, $distratado = null, $order = null, $aprovados = false, $contratosNaoQuitados = false)
    {
        $contrato = self::getInstance();

        $contratos = $contrato->getDefaultAdapter()->select()->from(array('c' => TB_CONTRATO), array('c.id as id', 'c.id_pessoa', 'c.fl_distrato'))
            ->join(array('l' => TB_LOTES), 'c.id_lote = l.id', array('l.id as id_lote', 'quadra', 'lote'))
            ->join(array('e' => TB_EMPREENDIMENTO), 'l.id_empreendimento = e.id', array('e.id as id_empreendimento'))
            ->join(array('p' => TB_PESSOA), 'p.id = c.id_pessoa', array('p.nm_pessoa'))
            ->join(array('b' => TB_BANCO), 'b.id = c.id_banco', array('b.cd_banco'))
            ->joinLeft(array('agr' => TB_AGRUPADA), 'agr.idTabela = 9 AND agr.idCampo = b.cd_banco', array('nmBanco' => 'agr.descricao'));

        if ($contratosNaoQuitados) {
            $contratos = $contratos->join(array('pa' => TB_PARCELA), 'c.id = pa.id_contrato', '')
                ->where('pa.pago != "1"')
                ->group('c.id');
        }
        
        $contratos = $contratos->where('e.id = ?', $empreendimento);

        if ($banco)
            $contratos = $contratos->where('b.id = ?', $banco);

        if ($order)
            $contratos = $contratos->order($order);

        //testa o parametro do tipo de contrato (gerado parcelas ou no)
        if ($tipo != self::CONTRATO_TODOS && $tipo != null)
            $contratos = $contratos->where('c.fl_gerado = ?', $tipo);

        //testa se vai procurar os distratados ou no (default retorna todos)
        if ($distratado)
            $contratos = $contratos->where("c.fl_distrato = '0'");

        if ($aprovados)
            $contratos = $contratos->where("c.fl_aprovar_contrato = '1'");

        $contratos = $contratos->query()->fetchAll();

        return $contratos;
    }

    /**
     * Retorna o id de um contrato no distratado pelo lote relacionado
     * @param $lote
     * @return bool
     */
    public static function getIdByLote($lote, $full = false)
    {
        $contrato = self::getInstance();
        $contrato = $contrato->getDefaultAdapter()->select()->from(array('c' => TB_CONTRATO), '*')
            ->where('c.id_lote = ?', $lote)
            ->where('c.fl_distrato = "0"')
            ->query()->fetch();
        if (isset($contrato['id'])) {
            if (!$full)
                return $contrato['id'];
            else
                return $contrato;
        } else {
            return false;
        }
    }

    //procura o id dos lotes que possuem contrato pelo empreendimento, retorna ids e dados do contrato
    public static function loteByEmpreendimento($empreendimento, $tipo = self::CONTRATO_TODOS, $distratado = null, $order = null, $aprovados = true)
    {
        $contrato = self::getInstance();

        $contratos = $contrato->getDefaultAdapter()->select()->from(array('c' => TB_CONTRATO), array('id_lote'))
            ->join(array('l' => TB_LOTES), 'c.id_lote = l.id', array())
            ->join(array('e' => TB_EMPREENDIMENTO), 'l.id_empreendimento = e.id')
            ->join(array('p' => TB_PESSOA), 'p.id = c.id_pessoa', array('p.nm_pessoa'))
            ->where('e.id = ?', $empreendimento)
            ->where('c.fl_distrato = "0"');

        if ($aprovados)
            $contratos->where('c.fl_aprovar_contrato = "1"');

        if ($order)
            $contratos = $contratos->order($order);

        //testa o parametro do tipo de contrato (gerado parcelas ou no)
        if ($tipo != self::CONTRATO_TODOS && $tipo != null)
            $contratos = $contratos->where('c.fl_gerado = ?', $tipo);

        //testa se vai procurar os distratados ou no (default retorna todos)
        if ($distratado)
            $contratos = $contratos->where("c.fl_distrato = '0'");

        $contratos = $contratos->query()->fetchAll();

        return $contratos;
    }

    public static function byEmpreendimentoextrato($empreendimento, $tipo = self::CONTRATO_TODOS, $distratado = null, $order = null)
    {
        if (!$order)
            $order = "nm_pessoa";
        $contrato = self::getInstance();
        $select = $contrato->getAdapter()->select()->from(array('e' => TB_EMPREENDIMENTO), array())->
        join(array('l' => TB_LOTES), 'l.id_empreendimento = e.id', array('l.lote', 'l.quadra'))->
        join(array('c' => TB_CONTRATO), 'c.id_lote = l.id', '*')->where('e.id = ?', $empreendimento)->
        join(array('pes' => TB_PESSOA), 'pes.id = c.id_pessoa', array('nm_pessoa'));
        if ($tipo != self::CONTRATO_TODOS)
            $select->where('c.fl_gerado = ?', $tipo);

        if ($distratado) {
            if ($distratado != 'todos' || $distratado == true) {
                $select->where("c.fl_distrato IN ('0')");
            }
        } else {
            $select->where("c.fl_distrato IN ('1')");
        }

        $contratos = $select->order($order)->order('cast(l.lote AS UNSIGNED INTEGER) ASC')->query()->fetchAll();
        return $contratos;
    }

    public static function byEmpreendimentoParcelasAbertasHistorico($empreendimento, $tipo = self::CONTRATO_TODOS, $order = null)
    {
        if (!$order)
            $order = "pes.nm_pessoa";
        $contrato = self::getInstance();
        $select = $contrato->getAdapter()->select()->distinct('par.id_contrato')
            ->from(array('e' => TB_EMPREENDIMENTO), array())
            ->join(array('l' => TB_LOTES), 'l.id_empreendimento = e.id', array('lote', 'quadra'))
            ->join(array('c' => TB_CONTRATO), 'c.id_lote = l.id', '*')->where('e.id = ?', $empreendimento)
            ->join(array('ch' => TB_CONTRATO_HISTORICO), 'ch.id_lote = l.id', '*')
            ->join(array('pes' => TB_PESSOA), 'pes.id = c.id_pessoa', array('nm_pessoa'))
            ->join(array('pesC' => TB_PESSOA), 'pesC.id = ch.id_pessoa', array('nm_pessoa'))
            ->join(array('par' => TB_PARCELA), 'c.id = par.id_contrato', array('id_contrato'))
            ->join(array('parH' => TB_PARCELA_HISTORICO), 'ch.id = parH.id_contrato', array('id_contrato'))
            ->where('par.pago = ?', '0')
            ->where('parH.pago = ?', '0');
        if ($tipo != self::CONTRATO_TODOS)
            $select->where('c.fl_gerado = ?', $tipo);
        $select->where('ch.fl_gerado = ?', $tipo);
        return $select->order($order)->query()->fetchAll();
    }

    /**
     * Funo que buscar as informaes de parcelas abertas, pelo id_empredimento, pego no tb_lote.
     * @param $empreendimento
     * @param string $tipo
     * @param null $order
     * @param bool $fl_distrato
     * @return array
     */
    public static function byEmpreendimentoParcelasAbertas($empreendimento, $tipo = self::CONTRATO_TODOS, $order = null, $fl_distrato = false, $aprovados = false)
    {
        if (!$order)
        {
            $order = "pes.nm_pessoa";
        }
        
        $contrato = self::getInstance();
        
        $select = $contrato->getAdapter()->select()->distinct('par.id_contrato')
            ->from(array('c' => TB_CONTRATO), array('*'))
            ->join(array('l' => TB_LOTES), 'l.id = c.id_lote', array('lote', 'quadra'))
            ->join(array('pes' => TB_PESSOA), 'pes.id = c.id_pessoa', array('nm_pessoa'))
            ->join(array('par' => TB_PARCELA), 'c.id = par.id_contrato', array('id_contrato'))
            ->where('l.id_empreendimento = ?', $empreendimento)
            ->where('par.pago = ?', '0');
        
        if ($tipo != self::CONTRATO_TODOS)
        {
            $select->where('c.fl_gerado = ?', $tipo);
        }
        
        if (!$fl_distrato)
        {
            $select = $select->where('c.fl_distrato = "0"');
        }
        else {
            $select = $select->where('c.fl_distrato = "1"');
        }

        if ($aprovados)
        {
            $select = $select->where("c.fl_aprovar_contrato = '1'");
        }
        return $select->order($order)->query()->fetchAll();
    }
    /**
     * Busca os contratos Distratados de acordo com o id do empreendimento
     */
    public static function byEmpreendimentoContratosDistratados($idEmpreendimento, $tipo = self::CONTRATO_TODOS, $order = null, $flDistrato = false)
    {
        if (!$order)
        {
            $order = "pes.nm_pessoa";
        }
        
        $contrato = self::getInstance();
        
        $select = $contrato->getAdapter()->select()->distinct('par.id_contrato')
            ->from(array('c' => TB_CONTRATO), array('*'))
            ->join(array('l' => TB_LOTES), 'l.id = c.id_lote', array('lote', 'quadra'))
            ->join(array('pes' => TB_PESSOA), 'pes.id = c.id_pessoa', array('nm_pessoa'))
            ->join(array('par' => TB_PARCELA_HISTORICO), 'c.id = par.id_contrato', array('id_contrato'))
            ->where('l.id_empreendimento = ?', $idEmpreendimento)
            ->where('par.pago = ?', '0');
        
        if ($tipo != self::CONTRATO_TODOS)
        {
            $select->where('c.fl_gerado = ?', $tipo);
        }
        
        if (!$flDistrato)
        {
            $select = $select->where('c.fl_distrato = "0"');
        }
        else {
            $select = $select->where('c.fl_distrato = "1"');
        }
        return $select->order($order)->query()->fetchAll();
    }

    public static function byEmpreendimentoParcelasAbertasQuitacao($empreendimento, $tipo = self::CONTRATO_TODOS, $order = null)
    {
        if (!$order)
            $order = "pes.nm_pessoa";
        $contrato = self::getInstance();
        $select = $contrato->getAdapter()->select()->distinct('par.id_contrato')->from(array('e' => TB_EMPREENDIMENTO), array())
            ->join(array('l' => TB_LOTES), 'l.id_empreendimento = e.id', array('lote', 'quadra'))
            ->join(array('c' => TB_CONTRATO), 'c.id_lote = l.id', '*')->where('e.id = ?', $empreendimento)
            ->join(array('pes' => TB_PESSOA), 'pes.id = c.id_pessoa', array('nm_pessoa'))
            ->join(array('par' => TB_PARCELA), 'c.id = par.id_contrato', array('id_contrato'))
            ->where('par.pago = ?', '0')
            ->where('par.tp_parcela != ?', 'Q');

        if ($tipo != self::CONTRATO_TODOS) {
            $select->where('c.fl_gerado = ?', $tipo);
        }
        return $select->order($order)->query()->fetchAll();
    }

    public static function hasParcela($id_contrato)
    {
        $parcelas = new Parcela();
        $quant = $parcelas->select()->from(TB_PARCELA, 'count(*) AS quant')->where('id_contrato = ?', $id_contrato)->query()->fetchAll();
        return ($quant[0]['quant'] > 0);
    }

    public static function getNextMonth($data, $level = 1, $monthInc = true)
    {
        $data = explode('-', $data);
        $data['year'] = (int)$data[0];
        $data['month'] = (int)$data[1];
        $data['day'] = (int)$data[2];

        if (!$data['year'] || !$data['month'] || !$data['day'])
            return false;

        if ($monthInc)
            $data['month'] = $data['month'] + $level;
        while ($data['month'] > 12) {
            $data['month'] = $data['month'] - 12;
            $data['year'] = $data['year'] + 1;
        }

        if (checkdate($data['month'], $data['day'], $data['year'])) {
            $month = str_pad($data['month'], 2, '0', STR_PAD_LEFT);
            $day = str_pad($data['day'], 2, '0', STR_PAD_LEFT);
            $year = str_pad($data['year'], 2, '0', STR_PAD_LEFT);
            return $year . '-' . $month . '-' . $day;
        } else {
            $data['day'] = $data['day'] - 1;
            $d = $data['year'] . '-' . $data['month'] . '-' . $data['day'];
            return self::getNextMonth($d, $level, false);
        }
    }

    public static function getNextWeek($data, $adicional = 1, $next_dt_normal, $weekInc = true)
    {
        if($next_dt_normal){
            $data = explode('-', $next_dt_normal);
        }else{
            $weekInc = false;
            $data = explode('-', $data);
        }

        $data['year'] = (int)$data[0];
        $data['month'] = (int)$data[1];
        $data['day'] = (int)$data[2];

        if (!$data['year'] || !$data['month'] || !$data['day'])
            return false;
       
        if ($weekInc)
            $data['day'] = $data['day'] + 7;
        if($data['month'] == 4 || $data['month'] == 6 || $data['month'] == 9 || $data['month'] == 11){ //Possui 30 dias 
            while ($data['day'] > 30) {
                $data['day'] = $data['day'] - 30;
                $data['month'] = $data['month'] + 1;
            }
        }elseif($data['month'] == 1 || $data['month'] == 3 || $data['month'] == 5 || $data['month'] == 7 || $data['month'] == 8 || $data['month'] == 10 || $data['month'] == 12){ //Possui 31 dias
            while ($data['day'] > 31) {
                $data['day'] = $data['day'] - 31;
                $data['month'] = $data['month'] + 1;
            }
        }else{ //Fevereiro (28 ou 29 dias)
            if($data['year']%4 == 0){ //Se  bissexto
                while ($data['day'] > 29) {
                    $data['day'] = $data['day'] - 29;
                    $data['month'] = $data['month'] + 1;
                }
            }else{//Se no  bissexto
                while ($data['day'] > 28) {
                    $data['day'] = $data['day'] - 28;
                    $data['month'] = $data['month'] + 1;
                }
            }
        }
        while ($data['month'] > 12) {
            $data['month'] = $data['month'] - 12;
            $data['year'] = $data['year'] + 1;
        }

        if (checkdate($data['month'], $data['day'], $data['year'])) {
            $month = str_pad($data['month'], 2, '0', STR_PAD_LEFT);
            $day = str_pad($data['day'], 2, '0', STR_PAD_LEFT);
            $year = str_pad($data['year'], 2, '0', STR_PAD_LEFT);
            return $year . '-' . $month . '-' . $day;
        } else {
            $data['day'] = $data['day'] - 1;
            $d = $data['year'] . '-' . $data['month'] . '-' . $data['day'];
            return self::getNextMonth($d, $level, false);
        }
    }

    public static function getParcelas($dtInicialParcela, $qtdParcela, $frequencia = 1)
    {

        if (!$dtInicialParcela && !$qtdParcela) {
            return false;
        } else if ($frequencia < 1) {
            return false;
        }

        if (strstr($dtInicialParcela, '/')) {
            $dtParcelaNext = Helper::getInputDate($dtInicialParcela);
        }
        if (strstr($dtInicialParcela, '-')) {
            $dtParcelaNext = $dtInicialParcela;
        }

        for ($i = 0; $i <= $qtdParcela; $i++) {
            if ($i != 0) {
                $dtParcelaNext = Contrato::getNextMonth($dtParcelaNext, $frequencia);
            }
            $arrayParcelas[$i] = $dtParcelaNext;
        }

        return $arrayParcelas;
    }

    /**
     * @return Contrato
     */
    public static function getInstance()
    {
        if (!self::$contratoInstacia) {
            self::$contratoInstacia = new Contrato();
        }
        return self::$contratoInstacia;
    }

    /**
     * @param int $id_contrato
     * @@return void
     */
    public static function gerarParcela($id_contrato)
    {
        $contratos = self::getInstance();
        $contrato = $contratos->fetchRow('id = '.$id_contrato);

        //Calcula a quantidade de parcelas a serem geradas
        $qtdNormais = (int)$contrato->nr_parcela;
        $qtdIntercaladas = (int)$contrato->nr_intercalada;
        $qtdChaves = (int)$contrato->nr_parcela;

        //Busca tipos de parcelas adicionais de cada tipo
        if ($contrato->fl_itens_contrato == 1) {
            $itens = new ContratoItens();
            $itens = $itens->fetchAll('id_contrato = ' . $contrato->id);

            $auxItens = array();
            if (count($itens) > 0)
                foreach ($itens as $item) {
                    $auxItens[$item['tp_parcela']][] = $item->toArray();
                }
            $itens = $auxItens;

            if (count($itens['N']) > 0)
                foreach ($itens['N'] as $adicionais) {
                    $qtdNormais += (int)$adicionais['qt_parcelas'];
                }
            if (count($itens['I']) > 0)
                foreach ($itens['I'] as $adicionais) {
                    $qtdIntercaladas += (int)$adicionais['qt_parcelas'];
                }
            if (count($itens['C']) > 0)
                foreach ($itens['C'] as $adicionais) {
                    $qtdChaves += (int)$adicionais['qt_parcelas'];
                }
        }

        //PARCELAS INTERCALADAS
        $primeira_dt_intercalada = $contrato->dt_intercalada;

        //Array que vai receber as parcelas intercaladas
        $parcelasIntercaladas = array();

        //$ii: valor Incremento das parcelas Intercaladas
        for ($ii = 0; $ii < $contrato->nr_intercalada; $ii++) {
            $next_dt_interc = self::getNextMonth($primeira_dt_intercalada, $contrato->fr_intercalada * $ii);

            //Cria o array com os dados da parcela"
            $dados['parcela'] = (float)$contrato->vl_intercalada;
            $dados['dt_parcela'] = $next_dt_interc;
            $dados['tp_parcela'] = 'I';
            $dados['fl_reajustavel'] = $contrato->fl_reajustavel_intercaladas;

            $parcelasIntercaladas[] = $dados;
        }

        //Testa se possui itens adicionais
        if ($contrato->fl_itens_contrato == 1) {
            if (count($itens['I']) > 0)
                //Continua a interao da mesma forma das parcelas anteriores mas agora com as adicionais
                foreach ($itens['I'] as $item) {
                    for ($i = 0; $i < $item['qt_parcelas']; $i++) {
                        if ($item['dt_parcela'])
                            $dt_intercalada = $item['dt_parcela'];
                        else
                            $dt_intercalada = $primeira_dt_intercalada;

                        if ($item['nr_frequencia'])
                            $frequencia = $item['nr_frequencia'];
                        else
                            $frequencia = $contrato->fr_intercalada;

                        $next_dt_interc = self::getNextMonth($dt_intercalada, $frequencia * $i);

                        //Cria o array com os dados da parcela"
                        $dados['parcela'] = (float)$item['vl_parcela'];
                        $dados['dt_parcela'] = $next_dt_interc;
                        $dados['tp_parcela'] = 'I';
                        $dados['fl_reajustavel'] = $item['fl_reajustavel'];

                        $parcelasIntercaladas[] = $dados;
                    }
                }
        }

        //PARCELAS NORMAIS
        $primeira_dt_normal = $contrato->dt_primeira_parcela;

        //Array que vai receber as parcelas normais
        $parcelasNormais = array();

        //Valor de Incremento das parcelas Normais
        $in = 0;
        for ($i = 0; $i < $contrato->nr_parcela; $i++) {
            if($contrato->fl_frequencia == 'S'){
                $next_dt_normal = self::getNextWeek($primeira_dt_normal, $in, $next_dt_normal);
            }else{
                $next_dt_normal = self::getNextMonth($primeira_dt_normal, $in);
            }

            //Se no puder coincidir data de normais e intercaladas, retorna data da parcela normal para o ms seguinte
            if ($contrato['fl_coinc_intercalada'] == 0 && $parcelasIntercaladas) {
                $next_dt_normal_info = self::isDateCoincident($parcelasIntercaladas, $primeira_dt_normal, $next_dt_normal, $in);
                $next_dt_normal = $next_dt_normal_info["dt"];
                $in = $next_dt_normal_info["in"];
            }
            //Calculo para contratos que possui o SACOC
            if ($contrato['tx_juros_sacoc'] && $contrato['nr_meses_financiamento_sacoc'] && $contrato['tx_juros_sacoc']) {
                if ($i == 0) {
                    $dados['parcela'] = (float)$contrato->vl_parcela;
                } else {
                    $dados['parcela'] = ($dados['parcela'] * $contrato['tx_juros_sacoc'] / 100) + $dados['parcela'];
                }
            } else {
                $dados['parcela'] = (float)$contrato->vl_parcela;
            }

            //Cria o array com os dados da parcela"
            $dados['parcela'] = $dados['parcela'];
            $dados['dt_parcela'] = $next_dt_normal;
            $dados['tp_parcela'] = 'N';
            $dados['fl_reajustavel'] = $contrato->fl_reajustavel_mensais;
            $in++;

            $parcelasNormais[] = $dados;
        }


        //Testa se possui itens adicionais
        if ($contrato->fl_itens_contrato == 1) {
            if (count($itens['N']) > 0)
                //Continua a interao da mesma forma das parcelas anteriores mas agora com as adicionais
                foreach ($itens['N'] as $item) {
                    for ($i = 0; $i < $item['qt_parcelas']; $i++) {
                        if($contrato->fl_frequencia == 'S'){
                            $next_dt_normal = self::getNextWeek($primeira_dt_normal, $in, $next_dt_normal);
                        }else{
                            $next_dt_normal = self::getNextMonth($primeira_dt_normal, $in);
                        }
                        

                        //Se no puder coincidir data de normais e intercaladas, retorna data da parcela normal para o ms seguinte
                        if ($contrato['fl_coinc_intercalada'] == 0 && $parcelasIntercaladas) {
                            $next_dt_normal_info = self::isDateCoincident($parcelasIntercaladas, $primeira_dt_normal, $next_dt_normal, $in);
                            $next_dt_normal = $next_dt_normal_info["dt"];
                            $in = $next_dt_normal_info["in"];
                        }

                        //Cria o array com os dados da parcela"
                        $dados['parcela'] = (float)$item['vl_parcela'];
                        $dados['dt_parcela'] = $next_dt_normal;
                        $dados['tp_parcela'] = 'N';
                        $dados['fl_reajustavel'] = $item['fl_reajustavel'];

                        $parcelasNormais[] = $dados;
                        $in++;
                    }
                }
        }


        // PARCELAS SINAL
        $tb_sinal = new Sinal();
        $parcsSinal = $tb_sinal->fetchAll("idContrato = " . $id_contrato);

        //Array que vai receber as parcelas sinal
        $parcelasSinal = array();
        foreach ($parcsSinal as $sinal) {
            //Cria o array com os dados da parcela"
            $dados['parcela'] = (float)$sinal->valor;
            $dados['dt_parcela'] = $sinal->dtVencimento;
            $dados['tp_parcela'] = 'S';
            $dados['fl_reajustavel'] = $contrato->fl_reajustavel_chaves;

            $parcelasSinal[] = $dados;
        }

        // PARCELAS CHAVES
        $dt_parcela_chave = $contrato->dt_parcela_entrega;
        $qtdParcelas = $contrato->nr_parcela_entrega;

        //Array que vai receber as parcelas chave
        $parcelasChave = array();

        for ($i = 0; $i < $qtdParcelas; $i++) {
            // Calcula a proxima data
            $next_dt_chave = self::getNextMonth($dt_parcela_chave, $i);

            //Cria o array com os dados da parcela"
            $dados['parcela'] = (float)$contrato->vl_parcela_entrega;
            $dados['dt_parcela'] = $next_dt_chave;
            $dados['tp_parcela'] = 'C';

            $parcelasChave[] = $dados;
        }
        //Ajusta o contador das parcelas chave para remover o ultimo indice que saiu do for
        $ic = $i - 1;

        //Testa se possui itens adicionais
        if ($contrato->fl_itens_contrato == 1) {
            if (count($itens['C']) > 0)
                //Continua a interao da mesma forma das parcelas anteriores mas agora com as adicionais
                foreach ($itens['C'] as $item) {
                    for ($i = 0; $i < $item['qt_parcelas']; $i++) {
                        $ic++;

                        // Calcula a proxima data
                        $next_dt_chave = self::getNextMonth($dt_parcela_chave, $ic);

                        //Cria o array com os dados da parcela"
                        $dados['parcela'] = (float)$item['vl_parcela'];
                        $dados['dt_parcela'] = $next_dt_chave;
                        $dados['tp_parcela'] = 'C';
                        $dados['fl_reajustavel'] = $item['fl_reajustavel'];

                        $parcelasChave[] = $dados;
                    }
                }
        }

        //Array que recebe todas as parcelas separadas por tipo
        $parcelas = array(
            $parcelasIntercaladas,
            $parcelasNormais,
            $parcelasSinal,
            $parcelasChave
        );

        foreach ($parcelas as $parcelasTipo) {

            if (count($parcelasTipo) > 0) {

                //Ordena parcelas por data e tipo
                ksort($parcelasTipo);

                //(Re)Inicia a contagem da id da parcela
                $id = 1;

                //salvar parcelas
                foreach ($parcelasTipo as $data => $parcela) {
                    $saveParcela = new Parcela();
                    $newParcela = $saveParcela->createRow();
                    $newParcela->id_parcela = $id;
                    $newParcela->id_contrato = $id_contrato;
                    $newParcela->acrescimo = $contrato->acrescimo;
                    $newParcela->desconto = $contrato->desconto;
                    $newParcela->vl_parcela = $parcela['parcela'];
                    $newParcela->dt_parcela = $parcela['dt_parcela'];
                    $newParcela->tp_parcela = $parcela['tp_parcela'];
                    $newParcela->fl_reajustavel = $parcela['fl_reajustavel'];
                    $newParcela->pago = '0';
                    $newParcela->fl_envio = '0';
                    $newParcela->nr_tipo_parcela = $id."/".count($parcelasTipo)." ".$parcela['tp_parcela'];
                    $newParcela->obs = '';
                    $newParcela->log = (Login::getUsuario()->getLogin() . ' - ' . date('d/m/Y H:m:s') . ' - i');

                    try {
                        $newParcela->save();
                        $id++;
                    } catch (Exception $e) {
                        die($e->getMessage());
                    }
                }
            }

            $contratos->update(array('fl_gerado' => '1',), $contratos->getAdapter()->quoteInto('id = ?', $id_contrato));
        }
    }

    /**
     *
     * @param string $termo
     * @return array
     */
    public static function fullSearch($termo, $count = false, $offset = false, $page = false, $where = null)
    {
        $result = array();
        $termo = "'%" . strtoupper($termo) . "%'";
        $db = DB::getInstance();

        if ($count && $offset === false && $page === false) {
            $select = "SELECT count(c.id) AS count FROM " . TB_CONTRATO . " c
                    JOIN " . TB_PESSOA . " p  ON c.id_pessoa = p.id
                    JOIN " . TB_LOTES . " l  ON c.id_lote = l.id
                    JOIN " . TB_EMPREENDIMENTO . " e  ON l.id_empreendimento = e.id
                  WHERE CONCAT(UPPER(e.nm_empreendimento),' ',c.id,' ',l.lote,' ',UPPER(p.nm_pessoa)) LIKE {$termo}";
            if ($where) {
                $select .= " AND $where";
            }
        } else {
            $select = "SELECT c.id FROM " . TB_CONTRATO . " c
                    JOIN " . TB_PESSOA . " p  ON c.id_pessoa = p.id
                    JOIN " . TB_LOTES . " l  ON c.id_lote = l.id
                    JOIN " . TB_EMPREENDIMENTO . " e  ON l.id_empreendimento = e.id
                  WHERE CONCAT(UPPER(e.nm_empreendimento),' ',c.id,' ',l.lote,' ',UPPER(p.nm_pessoa)) LIKE {$termo} ";
            if ($where) {
                $select .= " AND $where";
            }
            if ($offset !== false && $page !== false)
                $select .= " LIMIT {$offset}, {$page} ";
        }
        $stmt = $db->prepare($select);
        $stmt->execute();
        $ids = $stmt->fetchAll(PDO::FETCH_ASSOC);

        if ($count && $offset === false && $page === false)
            return $ids;

        if (is_array($ids) && count($ids)) {
            foreach ($ids as $id) {
                $result[] = self::findContrato($id['id']);
            }
        }
        return $result;
    }

    /**
     *
     * @param integer $id
     * @return array
     */
    public static function findContrato($id)
    {
        $db = Db::getInstance();
        $stmt = $db->prepare("SELECT c.id, c.fl_aprovar_contrato, e.nm_empreendimento, e.id AS idEmpreendimento, l.lote, l.quadra, p.nm_pessoa FROM " . TB_CONTRATO . " c
                                JOIN " . TB_PESSOA . " p  ON c.id_pessoa = p.id
                                JOIN " . TB_LOTES . " l  ON c.id_lote = l.id
                                JOIN " . TB_EMPREENDIMENTO . " e  ON l.id_empreendimento = e.id
                              WHERE c.id = ?");
        $stmt->execute(array($id));
        $stmt->setFetchMode(PDO::FETCH_CLASS, "Contrato");
        return $stmt->fetch();
    }

    /**
     * Busca apenas os dados do contrato
     * @param $id
     * @param null $distratado
     * @return Contrato|Zend_Db_Select
     */
    public static function findById($id, $distratado = null)
    {
        $contrato = self::getInstance();

        $contrato = $contrato->getDefaultAdapter()->select()->from(array('c' => TB_CONTRATO), '*')
            ->where('c.id = ?', $id);

        //testa se vai procurar os distratados ou no (default retorna todos)
        if ($distratado)
            $contrato = $contrato->where("c.fl_distrato = '0'");

        $contrato = $contrato->query()->fetch();
        return $contrato;
    }

    /*
     * Funo para buscar contrato pelo ID do mesmo
     * @param string $id
     * @param boolean $distratado
     * @return obj contrato
     */
    public static function findContratoById($id, $distratado = null, $buscaJuridico = false,$empreendimento = false)
    {
        $contrato = self::getInstance();

        $contrato = $contrato->getDefaultAdapter()->select()->from(array('c' => TB_CONTRATO), '*')
            ->join(array('p' => TB_PESSOA), 'p.id = c.id_pessoa', array('*', 'idPessoa' => 'p.id'))
            ->joinLeft(array('pf' => TB_PESSOA_FISICA), 'p.id = pf.id_pessoa', '*')
            ->joinLeft(array('pj' => TB_PESSOA_JURIDICA), 'p.id = pj.id_pessoa', '*')
            ->joinLeft(array('j' => TB_JURIDICO), 'c.id = j.id_contrato', 'id_juridico')
            ->joinLeft(array('ci' => TB_CONTRATO_ITENS), 'c.id = ci.id_contrato', array('ci.vl_parcela as vl_parcela_adicional', 'qt_parcelas'))
            ->join(array('l' => TB_LOTES), 'l.id = c.id_lote', '*');

        if($empreendimento){
            $contrato = $contrato->join(array('e' => TB_EMPREENDIMENTO), 'l.id_empreendimento = e.id', array('nm_empreendimento','cnpjContato','telefoneContato','emailContato'))
            ->joinLeft(array('tl' => TB_TIPO_LOTEAMENTO), 'e.id_tipo_loteamento = tl.id_tipo_loteamento', array('ds_tipo_loteamento', 'ds_grupo', 'ds_subgrupo'));
        }
        if ($buscaJuridico)
        {
            $contrato = $contrato->joinLeft(array('ag' => TB_AGRUPADA), 'j.id_tipo_audiencia = ag.idCampo AND ag.idTabela = "18"', 'descricao');
        }
        
        $contrato = $contrato->where('c.id = ?', $id);

        //testa se vai procurar os distratados ou no (default retorna todos)
        if ($distratado)
            $contrato = $contrato->where("c.fl_distrato = '0'");

        $contrato = $contrato->query()->fetch();
        return $contrato;
    }

    /*
     * Funo para buscar contrato pelo ID do mesmo apenas com dados para relatorio simples
     * @param string $id
     * @param boolean $distratado
     * @return obj contrato
     */
    public static function findContratoByIdRelatorio($id, $distratado = null)
    {
        $contrato = self::getInstance();

        $contrato = $contrato->getDefaultAdapter()->select()->from(array('c' => TB_CONTRATO), array('id_contrato' => 'c.id',
            'id_lote' => 'c.id_lote', 'nr_parcela', 'nr_parcela_sinal', 'nr_intercalada', 'nr_parcela_entrega'))
            ->join(array('l' => TB_LOTES), 'c.id_lote = l.id', array('l.quadra', 'l.lote'))
            ->join(array('p' => TB_PESSOA), 'c.id_pessoa = p.id', array('p.nm_pessoa', 'p.endereco', 'p.nm_bairro', 'id_pessoa' => 'p.id'))
            ->joinLeft(array('pf' => TB_PESSOA_FISICA), 'p.id = pf.id_pessoa', array('nr_cpf'))
            ->where('c.id = ?', $id);

        //testa se vai procurar os distratados ou no (default retorna todos)
        if ($distratado)
            $contrato = $contrato->where("c.fl_distrato = '0'");

        $contrato = $contrato->query()->fetch();
        return $contrato;
    }

    /*
     * Funo para buscar contrato pelo ID do mesmo
     * @param string $id
     * @param boolean $distratado
     * @return obj contrato
     */
    public static function findByIdPessoa($idPessoa, $distratado = null)
    {
        $contrato = self::getInstance();

        $contrato = $contrato->getDefaultAdapter()->select()->from(array('c' => TB_CONTRATO), '*')
            ->join(array('p' => TB_PESSOA), 'p.id = c.id_pessoa', array('*', 'idPessoa' => 'p.id'))
            ->joinLeft(array('pf' => TB_PESSOA_FISICA), 'p.id = pf.id_pessoa', '*')
            ->joinLeft(array('pj' => TB_PESSOA_JURIDICA), 'p.id = pj.id_pessoa', '*')
            ->join(array('l' => TB_LOTES), 'l.id = c.id_lote', '*')
            ->where('c.id_pessoa = ?', $idPessoa);

        //testa se vai procurar os distratados ou no (default retorna todos)
        if ($distratado)
            $contrato = $contrato->where("c.fl_distrato = '0'");

        $contrato = $contrato->query()->fetch();
        return $contrato;
    }

    /*
     * Funo para buscar contrato pelo ID do mesmo
     * @param string $id
     * @param boolean $distratado
     * @return obj contrato
     */
    public static function findByIdCorretor($idCorretor, $idEmpreendimento = false)
    {
        $contrato = self::getInstance();
        $contratos = $contrato->getDefaultAdapter()->select()
            ->from(array('c' => TB_CONTRATO), '*')
            ->join(array('l' => TB_LOTES), 'c.id_lote = l.id', array(''))
            ->where('c.id_corretor = ?', $idCorretor);

        if ($idEmpreendimento)
            $contratos = $contratos->where('l.id_empreendimento = ?', $idEmpreendimento);

        $contratos = $contratos->query()->fetchAll();
        return $contratos;
    }

    /*
     * Varre as parcelas dos contratos para buscar os contratos com parcelas do tipo quitao (Q) no pagas
     * @return array
     */
    public static function findParcelaQuitacao($post, $count = false)
    {
        $contratos = self::getDefaultAdapter()->select()
            ->from(array('c' => TB_CONTRATO), '*')
            ->join(array('p' => TB_PESSOA), 'c.id_pessoa = p.id', 'p.nm_pessoa')
            ->join(array('l' => TB_LOTES), 'c.id_lote = l.id', array('l.lote', 'l.quadra'))
            ->join(array('e' => TB_EMPREENDIMENTO), 'l.id_empreendimento = e.id', 'e.nm_empreendimento')
            ->join(array('pa' => TB_PARCELA), 'c.id = pa.id_contrato')
            ->where('pa.tp_parcela = "Q"')
            ->order('pa.id_renegociacao'); //Removido where que selecionava apenas um empreendimento

        //verifica se o usuario tem o perfil de usuario de empreendimento
        if ($_SESSION['usuario']) {
            try {
                $instanceUsuarioEmpreendimento = new Usuario_Empreendimento();
                $ids = $instanceUsuarioEmpreendimento->listaEmpreendimentosRepresentante($_SESSION['usuario']);
                if (count($ids)) {
                    $id = implode(',', $ids);
                    $contratos->where("e.id IN({$id})");
                }
            } catch (Exception $e) {

            }
        }

        if (!$count)
            if (isset($post['start']) && $post['length'] != '-1')
                $contratos = $contratos->limit($post['length'], $post['start']);
                
        $contratos = $contratos->group('c.id')->query()->fetchAll();
        return $contratos;
    }

    /*
     * Busca as parcelas do contrato e calcula seu valor total pago
     * @param integer $id
     * @return float $valorPago
     */
    public static function getParcelasValorById($id, $comAvulsa = true, $comSinal = null)
    {
        $parcelas = Parcela::findParcelasPagasById($id);
        $valorPago = 0;

        foreach ($parcelas as $parcela) {
            
            if ($comSinal == '0' && $parcela['tp_parcela'] == 'S')
                continue;

            if ($comAvulsa) {
                $valorPago += $parcela['vl_parcela'] - $parcela['desconto'] + $parcela['acrescimo'];
            } else if ($parcela['tp_parcela'] != 'A') {
                $valorPago += $parcela['vl_parcela'] - $parcela['desconto'] + $parcela['acrescimo'];
            }
        }

        return $valorPago;
    }

    /**
     * Retorna o valor total do contrato de acordo com os dados cadastrais (sem considerar as parcelas)
     * @param $id
     * @return float|int
     */
    public static function getVlContrato($id)
    {
        $contrato = self::getInstance()->fetchRow('id = ' . $id);

        $valorTotal = ((int)$contrato->nr_parcela * (float)$contrato->vl_parcela)
            + ((int)$contrato->nr_intercalada * (float)$contrato->vl_intercalada)
            + ((int)$contrato->nr_parcela_entrega * (float)$contrato->vl_parcela_entrega);
        if ($contrato->inclui_sinal_contrato == 1) {
            $valorTotal += (float)$contrato->vl_sinal;
        }

        if ($contrato->fl_itens_contrato == '1') {
            $itensContrato = ContratoItens::getInstance()->fetchAll('id_contrato = ' . $id);
            foreach ($itensContrato as $item) {
                $valorTotal += (int)$item['qt_parcelas'] * (float)$item['vl_parcela'];
            }
        }
        
        return $valorTotal;
    }

    /**
     * Varre toda a array atual de parcelas procurando datas coincidentes  $dt_normal
     * Se encontrar datas coincidentes, incrementa o ms de $dt_normal
     * S retorna a data quando todas as datas das parcelas forem comparadas
     *
     * @param integer $parcelas
     * @param integer $primeira_dt_normal
     * @param integer $next_dt_normal
     * @param integer $in
     * @return array
     */
    public static function isDateCoincident($parcelas, $primeira_dt_normal, $next_dt_normal, $in)
    {
        $next_month_normal = explode('-', $next_dt_normal)[1];
        $next_year_normal = explode('-', $next_dt_normal)[0];

        foreach ($parcelas as $dt => $vl) {
            $next_month_dt = explode('-', $vl['dt_parcela'])[1];
            $next_year_dt = explode('-', $vl['dt_parcela'])[0];
            if ($next_month_normal == $next_month_dt && $next_year_normal == $next_year_dt) {
                $in++;
                $next_dt_normal = self::getNextMonth($primeira_dt_normal, $in);
                if ($vl == end($parcelas)) {
                    return array("dt" => $next_dt_normal, "in" => $in);
                } else {
                    $next_month_normal = explode('-', $next_dt_normal)[1];
                    $next_year_normal = explode('-', $next_dt_normal)[0];
                }
            }
        }

        return array("dt" => $next_dt_normal, "in" => $in);
    }

    public static function findLista($post, $count = false)
    {
        $select = "*";
        if($count){
            $select = "COUNT(*) as contador";
        }
        $search = utf8_decode($_POST['search']['value']);

        $order = [];
        if(isset($_POST['order'])){
            $order=$_POST['order'];
        }

        $array = array(
            0 => 'c.id',
            1 => 'c.nr_proposta',
            2 => 'e.nm_empreendimento',
            3 => 'l.quadra',
            4 => 'l.lote',
            5 => 'p.nm_pessoa'
        );

        if ($search) {
            $where = 'c.id LIKE "%' . $search . '%" ';
            $where .= 'or c.nr_proposta LIKE "%' . $search . '%" ';
            $where .= 'or e.nm_empreendimento LIKE "%' . $search . '%" ';
            $where .= 'or l.quadra LIKE "%' . $search . '%" ';
            $where .= 'or l.lote LIKE "%' . $search . '%" ';
            $where .= 'or p.nm_pessoa LIKE "%' . $search . '%" ';
            $where .= 'or cp_p.nm_pessoa LIKE "%' . $search . '%" ';
        }

        $contratos = self::getDefaultAdapter()->select()
            ->distinct()
            ->from(array('c' => TB_CONTRATO), array($select))
            ->join(array('p' => TB_PESSOA), 'c.id_pessoa = p.id', 'p.nm_pessoa')
            ->join(array('l' => TB_LOTES), 'c.id_lote = l.id', array('l.lote', 'l.quadra'))
            ->join(array('e' => TB_EMPREENDIMENTO), 'l.id_empreendimento = e.id', 'e.nm_empreendimento')
            ->joinleft(array('cp' => TB_CONTRATO_PESSOA), 'c.id = cp.id_contrato', array())
            ->joinleft(array('cp_p' => TB_PESSOA), 'cp.id_pessoa = cp_p.id', array())
            ->where('fl_distrato = "0"');
        if (isset($post['docusign']) ) {
            $contratos = $contratos->where('docusign_envelope_id IS NOT NULL');
            $contratos = $contratos->where('docusign_envelope_id != ""');
            if($post['docusign']=="1"){
                $contratos = $contratos->where('c.docusign_envelope_id in (select envelopeId from DOCUSIGN_EVENTOS where envelopeEvento="recipient-completed" group by envelopeId)');
            }
            elseif($post['docusign']=="0"){
                $contratos = $contratos->where('c.docusign_envelope_id not in (select envelopeId from DOCUSIGN_EVENTOS where envelopeEvento="recipient-completed" group by envelopeId)');
            }
        }
        if ($post['aprovar'] == '1') {
            $contratos = $contratos->where('fl_aprovar_contrato = "0"');
        }
        if($post['columns'][2]['search']['value']){
            $contratos = $contratos->where('e.nm_empreendimento LIKE ?',"%".$post['columns'][2]['search']['value']."%" );
        }
        if($post['columns'][3]['search']['value']){
            $contratos = $contratos->where('l.quadra LIKE ?',"%".$post['columns'][3]['search']['value'] ."%");
        }
        if($post['columns'][4]['search']['value']){
            $contratos = $contratos->where('l.lote LIKE ?',"%".$post['columns'][4]['search']['value']."%");
        }
        if($post['columns'][5]['search']['value']){
            $contratos = $contratos->where('p.nm_pessoa LIKE ?',"%".$post['columns'][5]['search']['value']."%" );
        }
        //verifica se o usuario tem o perfil de usuario de empreendimento
        if ($_SESSION['usuario']) {
            try {
                $instanceUsuarioEmpreendimento = new Usuario_Empreendimento();
                $ids = $instanceUsuarioEmpreendimento->listaEmpreendimentosRepresentante($_SESSION['usuario']);
                if (count($ids)) {
                    $id = implode(',', $ids);
                    $contratos->where("e.id IN({$id})");
                }
            } catch (Exception $e) {
            }
        }

        if (isset($where)) {
            $contratos = $contratos->where($where);
        }

        if (!$count){
            if (isset($post['start']) && $post['length'] != '-1') {
                $contratos = $contratos->limit($post['length'], $post['start']);
            }
        }
        $orderByQuery=[];
        foreach ($order as $ord)
            $orderByQuery[] = $array[$ord['column']] . ' ' . $ord['dir'];

        $contratos = $contratos->order(implode(',', $orderByQuery))->query()->fetchAll();

        return $contratos;
    }

    public static function contratos($idEmpreendimento = null) {

        $ano = date('Y');
        $mes = date('m');

        $contratosMesAtual = self::getDefaultAdapter()->select()
            ->from(array('c' => TB_CONTRATO), 'c.id', null)
            ->join(array('l' => TB_LOTES), 'c.id_lote = l.id', null)
            ->join(array('e' => TB_EMPREENDIMENTO), 'l.id_empreendimento = e.id', null)
            ->where("c.fl_distrato = '0'")
            ->where("c.fl_aprovar_contrato = '1'")
            ->where("dt_contrato <= '".$ano."-".$mes."-31'");

        if($idEmpreendimento)
            $contratosMesAtual = $contratosMesAtual->where("l.id_empreendimento = ?", $idEmpreendimento);

        return ['qtdMesAtual' => count(Helper::filtraPorUsuario($contratosMesAtual))];
    }

    //S MOSTRAR INADIMPLENTES PARA QUEM FOR SUPER USURIO.
    public static function inadimplentes($idEmpreendimento = null) {

        $ano = date('Y');
        $mes = date('m');
        $dia = date('d');

        $inadimplentesMesAtual = self::getDefaultAdapter()->select()
            ->from(array('c' => TB_CONTRATO), 'c.id, count(c.id) as contador')
            ->join(array('p' => TB_PARCELA), 'c.id = p.id_contrato', 'pago')
            ->join(array('l' => TB_LOTES), 'c.id_lote = l.id', null)
            ->join(array('e' => TB_EMPREENDIMENTO), 'l.id_empreendimento = e.id', null)
            ->join(array('pe' => TB_PESSOA), 'c.id_pessoa = pe.id', array('nm_pessoa'))
            ->where("dt_parcela < '".$ano."-".$mes."-".$dia."'")
            ->where("p.tp_parcela IN ('N', 'S', 'I', 'C', 'G', 'Q', 'A')")
            ->where("c.fl_distrato = '0'")
            ->group("c.id")
            ->group("pago");

        if($idEmpreendimento)
            $inadimplentesMesAtual = $inadimplentesMesAtual->where("l.id_empreendimento = ?", $idEmpreendimento);        

        $inadimplentesMesAtual = Helper::filtraPorUsuario($inadimplentesMesAtual);

        $parcPagas = 0;
        $totalParc = 0;
        foreach ($inadimplentesMesAtual as $iMA) {
            if($iMA['pago'] == '0'){
                $parcPagas += intval($iMA['contador']);
            };
            $totalParc+=intval($iMA['contador']);
        }

        return ['qtdMesAtual' => $parcPagas*100/$totalParc];
    }
    
    
     //ALTERAES RAILAN
    /**
 * Pega a pessoa e contrato pelo id do contrato
 * @param $idcontrato = id do contrato
 * @return array|Zend_Db_Select = lista dos contratos
 */
public static function getPessoaContratoEmpreendimento($idcontrato)
{
    $contrato = self::getInstance();
    $contratos = $contrato->getDefaultAdapter()->select()->from(array('c' => TB_CONTRATO), array('idcontrato'=>'c.id'))
        ->join(array('p' => TB_PESSOA), 'p.id = c.id_pessoa', array('idpessoa'=>'id','p.nm_pessoa'))
        ->joinLeft(array('pj' => TB_PESSOA_JURIDICA), 'pj.id_pessoa = p.id', array('nr_cnpj'))
        ->joinLeft(array('pf' => TB_PESSOA_FISICA), 'pf.id_pessoa = p.id', array('nr_cpf'))
        ->where('c.id = ?', $idcontrato);

    $contratos = $contratos->query()->fetch();
    return $contratos;

}

public static function getPessoaContratoEmpreendimentoPj($idcontrato)
{
    $contrato = self::getInstance();
    $contratos = $contrato->getDefaultAdapter()->select()->from(array('c' => TB_CONTRATO), array('idcontrato'=>'c.id'))
        ->join(array('p' => TB_PESSOA), 'p.id = c.id_pessoa', array('idpessoa'=>'id','p.nm_pessoa'))
        ->joinLeft(array('pj' => TB_PESSOA_JURIDICA), 'pj.id_pessoa = p.id', array('nr_cnpj'))
        ->joinLeft(array('pf' => TB_PESSOA_FISICA), 'pf.id_pessoa = p.id', array('nr_cpf'))
        ->joinLeft(array('r' => TB_REPRESENTANTE), 'r.id_pessoa_juridica = pj.id_pessoa', array('r.id_pessoa'))
        ->joinLeft(array('pr' => TB_PESSOA_FISICA), 'pr.id_pessoa = r.id_pessoa', array('nr_cpf'))
        ->where('c.id = ?', $idcontrato);

    $contratos = $contratos->query()->fetch();
    return $contratos;

}

public static function getAllContratosPj($empreendimento)
{
  $contrato = self::getInstance();
  $contratos = $contrato->getDefaultAdapter()->select()->from(array('c' => TB_CONTRATO), array('c.id as id', 'c.id_pessoa', 'c.fl_distrato'))
      ->join(array('l' => TB_LOTES), 'c.id_lote = l.id', array('l.id as id_lote', 'quadra', 'lote'))
      ->join(array('e' => TB_EMPREENDIMENTO), 'l.id_empreendimento = e.id', array('e.id as id_empreendimento'))
      ->join(array('p' => TB_PESSOA), 'p.id = c.id_pessoa', array('idpessoa'=>'id','p.nm_pessoa'))
      ->joinLeft(array('pj' => TB_PESSOA_JURIDICA), 'pj.id_pessoa = c.id_pessoa', array('pj.nr_cnpj'))
      ->joinLeft(array('r' => TB_REPRESENTANTE), 'r.id_pessoa_juridica = pj.id_pessoa', array('r.id_pessoa'))
      ->joinLeft(array('pr' => TB_PESSOA_FISICA), 'pr.id_pessoa = r.id_pessoa', array('nr_cpf'))
      ->where('p.id = pj.id_pessoa')
      ->where('c.fl_distrato = "0"')
      ->where('e.id = ?', $empreendimento);

    $contratos = $contratos->group('p.id')->query()->fetchAll();
  
  return $contratos;

}

public static function getAllContratos($empreendimento)
{
  $contrato = self::getInstance();
  $contratos = $contrato->getDefaultAdapter()->select()->from(array('c' => TB_CONTRATO), array('c.id as id', 'c.id_pessoa', 'c.fl_distrato'))
      ->join(array('l' => TB_LOTES), 'c.id_lote = l.id', array('l.id as id_lote', 'quadra', 'lote'))
      ->join(array('e' => TB_EMPREENDIMENTO), 'l.id_empreendimento = e.id', array('e.id as id_empreendimento'))
      ->join(array('p' => TB_PESSOA), 'p.id = c.id_pessoa', array())
      ->joinLeft(array('pf' => TB_PESSOA_FISICA), 'pf.id_pessoa = c.id_pessoa', array('pf.nr_cpf'))
      //->joinLeft(array('pj' => TB_PESSOA_JURIDICA), 'pj.id_pessoa = c.id_pessoa', array('pj.nr_cnpj'))
      ->where('p.id = pf.id_pessoa')
      ->where('c.fl_distrato = "0"')
      ->where('e.id = ?', $empreendimento);

    $contratos = $contratos->group('p.id')->query()->fetchAll();
  
  return $contratos;

}
public static function descontoQuitacao($data_Vencimento,$parcelaId)
{

    $dominio = parse_url($_SERVER['HTTP_REFERER'], PHP_URL_HOST);

    //$parcelaId = $_POST['parcelaId'];
    $parcela = new Parcela();
    $parcela = $parcela::parcelaById($parcelaId);

    //busca o banco atravz do id do contrato.
    $contrato = new Contrato();
    $contrato = $contrato->getContrato($parcela['id_contrato']);

    //Inserido alterao de encoding para exibir os campos
    $parcela['log'] = utf8_encode($parcela['log']);
    $parcela['obs'] = utf8_encode($parcela['obs']);

    //Checa se o contrato possui enquadramento na tabela Price
    $checkPrice = ($contrato['vl_divida'] && $contrato['nr_meses_financiamento'] && $contrato['tx_juros'])?true:false;
    if($checkPrice){

        //Condio de amortizao para o cliente souzaincormporacoes.
        if ($dominio == "souzaincorporacoes.acadeone.com.br") {

            $jurosmes = $contrato['tx_juros'] / 100;

            //Calculando o nmero de dias e converte em meses entre a data de Vencimento da Parcela e a data de pagamento (hoje).
            $firstDate  = new DateTime(Helper::dataParaAmericano($data_Vencimento));
            $secondDate = new DateTime(Helper::dataParaAmericano($parcela['dt_parcela']));
            $intvl = $firstDate->diff($secondDate);
            $nrdias = $intvl->days/30;
            $nrMes = intval($nrdias);

            //Calculando o valor de desconto da Tabela PRICE com base na quantidade de meses.
            $vl_parcela = $parcela['vl_parcela'];
            $vlrfinal = $vl_parcela*(1/pow((1+$jurosmes),$nrMes));

        }else{
            //Calculando o valor de juros ao dia
            $jurosdia = $contrato['tx_juros'] / 30; //Mes de 30 dias

            //Calculando o nmero de dias entre a data de Vencimento da Parcela e a data de pagamento (hoje).
            $firstDate  = new DateTime(Helper::dataParaAmericano($data_Vencimento));
            $secondDate = new DateTime(Helper::dataParaAmericano($parcela['dt_parcela']));
            $intvl = $firstDate->diff($secondDate);
            $nrdias = $intvl->days;


            //Calculando o valor de desconto da Tabela PRICE
            $vl_parcela = $parcela['vl_parcela'];
            $vlrfinal = $vl_parcela/pow(1+($jurosdia/100),$nrdias);
        }
        //Envia os dados via JSON para a tela de Boleto 2Via
        $vl_desconto = $vl_parcela - $vlrfinal;
        $dados['data_vencimento'] = $data_Vencimento;
        $dados['vlrfinal'] = $vlrfinal;
        $dados['vl_parcela'] = $parcela['vl_parcela'];
        $dados['desconto'] = round($vl_desconto,2);
        $dados['mensagem'] = "Calculado o valor do desconto com sucesso!";
        return $dados;
    }else{
        //Caso no se enquadre na tabela PRICE, retorna mensagem de erro.
        $dados['desconto'] = 0;
        $dados['mensagem'] = "Contrato nao se enquadra na tabela PRICE";
        return $dados;
    }
}

    public function buscaOutrosCompradoresDimob($contratos, $idRespJuridico, $empreendimento, $tipoGeracao, $pagamento, $anoContrato, $anoRecebimento, $outros, $data)
    {
        $contratos = implode('","', $contratos);

        $outrosCompradores = self::getDefaultAdapter()->select()
            ->from(['c' => TB_CONTRATO], ['id_contrato' => 'id', 'nr_parcela',
                'vl_parcela', 'nr_parcela_sinal', 'vl_sinal', 'nr_intercalada', 'vl_intercalada',
                'nr_parcela_entrega', 'vl_parcela_entrega', 'dt_contrato', 'inclui_sinal_contrato', 'fl_distrato', 'fl_itens_contrato'])
            ->join(['cp' => TB_CONTRATO_PESSOA], 'c.id = cp.id_contrato', ['id_pessoa', 'porcentagem_participacao_contrato'])
            ->join(array('l' => TB_LOTES), 'l.id = c.id_lote', ['lote', 'quadra'])
            ->join(array('pe' => TB_PESSOA), 'pe.id = cp.id_pessoa', array('idPessoa' => 'id', 'nm_pessoa', 'tp_pessoa'))
            ->joinLeft(array('pf' => TB_PESSOA_FISICA), 'pe.id = pf.id_pessoa', array('nr_cpf'))
            ->joinLeft(array('pj' => TB_PESSOA_JURIDICA), 'pe.id = pj.id_pessoa', array('nr_cnpj'))
            ->joinLeft(array('p' => TB_PARCELA), 'p.id_contrato = cp.id_contrato', array('tp_baixa','vl_total_pago','tp_parcela', 'pago', 'dt_pagamento', 'dt_credito'))
            ->joinLeft(array('ph' => TB_PARCELA_HISTORICO), 'ph.id_contrato = c.id', array('tp_baixa','vl_total_pago','tp_parcela', 'pago', 'dt_pagamento', 'dt_credito'))
            ->join(array('e' => TB_EMPREENDIMENTO), 'e.id = l.id_empreendimento', array('nm_empreendimento','idEmpreendimento'=>'id', 'nr_cep'))
            ->joinLeft(['sc' => TB_CIDADE], 'e.cd_cidade = sc.id', ['cdIbge' => 'sc.cd_ibge', 'ufIbge' => 'sc.uf'])
            ->where('c.id IN ("' . $contratos . '")');

            if ($idRespJuridico) {
                $outrosCompradores = $outrosCompradores->joinLeft(array('ec' => TB_EMPREENDIMENTO_CONTABEIS), 'ec.id_empreendimento = e.id', array('vl_porc_declarada'))
                    ->where('id_pessoa_juridica = '.$idRespJuridico);
            }

            if ($tipoGeracao == 'E') {
                $outrosCompradores = $outrosCompradores->where('e.id = ' . $empreendimento)
                    ->where("p.tp_parcela != 'A' or ph.tp_parcela != 'A'");
            } else {
                $outrosCompradores = $outrosCompradores->where('e.id IN (' . '"'.$empreendimento.'")')
                    ->where("p.tp_parcela != 'A' or ph.tp_parcela != 'A'");
            }

            if ($pagamento == 'nao') {

                if ($anoContrato) {
                    $outrosCompradores->where("p.pago = '1' or ph.pago = '1'")
                        ->where("YEAR(c.dt_contrato) = '{$anoContrato}'")
                        ->where("YEAR(p.".$data.") = {$anoRecebimento} or YEAR(ph.".$data.") = {$anoRecebimento}");
                } else {
                    $outrosCompradores->where("p.pago = '1' or ph.pago = '1'")
                        ->where("YEAR(p.".$data.") = {$anoRecebimento} or YEAR(ph.".$data.") = {$anoRecebimento}");
                }
            } else {

                if ($anoContrato) {
                    $outrosCompradores->where("YEAR(c.dt_contrato) = '{$anoContrato}'")
                        ->where("YEAR(p.".$data.") = '{$anoRecebimento}' or p.dt_pagamento is null or p.dt_credito is null or
                                      YEAR(ph.".$data.") = '{$anoRecebimento}' or ph.dt_pagamento is null or ph.dt_credito is null");
                } else {
                    $outrosCompradores->where("YEAR(p.".$data.") = '{$anoRecebimento}' or p.dt_pagamento is null or p.dt_credito is null or
                                            YEAR(ph.".$data.") = '{$anoRecebimento}' or ph.dt_pagamento is null or ph.dt_credito is null");
                }
            }

            if ($outros == '0') {
                $outrosCompradores->where("p.tp_baixa = 'T' or p.tp_baixa = 'B' or p.tp_baixa is null or
                                        ph.tp_baixa = 'T' or ph.tp_baixa = 'B' or ph.tp_baixa is null")
                    ->group("cp.id_pessoa");
            } else {
                $outrosCompradores->where("p.tp_baixa = 'T' or p.tp_baixa = 'B' or p.tp_baixa = 'O' or p.tp_baixa is null or
                                        ph.tp_baixa = 'T' or ph.tp_baixa = 'B' or ph.tp_baixa = 'O' or ph.tp_baixa is null")
                    ->group("cp.id_pessoa");
            }

        return $outrosCompradores->query()->fetchAll();
    }

    public function buscaOutrosCompradoresDimobNovo($contratos, $idRespJuridico)
    {
        $contratos = implode('","', $contratos);

        $contratosPF = Zend_Db_Table::getDefaultAdapter()->select()
            ->from(array('l' => TB_LOTES), array('lote', 'quadra'))
            ->join(array('c' => TB_CONTRATO), 'l.id = c.id_lote', array('nr_parcela',
                'vl_parcela', 'nr_parcela_sinal', 'vl_sinal', 'nr_intercalada', 'vl_intercalada',
                'nr_parcela_entrega', 'vl_parcela_entrega', 'dt_contrato', 'inclui_sinal_contrato', 'fl_distrato', 'fl_itens_contrato'))
            ->join(['cp' => TB_CONTRATO_PESSOA], 'c.id = cp.id_contrato', ['id_contrato', 'porcentagem_participacao_contrato'])
            ->join(array('pe' => TB_PESSOA), 'pe.id = cp.id_pessoa', array('nm_pessoa', 'idPessoa' => 'id', 'tp_pessoa'))
            ->join(array('pf' => TB_PESSOA_FISICA), 'pe.id = pf.id_pessoa', 'nr_cpf')
            ->joinLeft(array('pj' => TB_PESSOA_JURIDICA), 'pe.id = pj.id_pessoa', array('nr_cnpj'))
            ->join(array('e' => TB_EMPREENDIMENTO), 'e.id = l.id_empreendimento', array('nm_empreendimento','idEmpreendimento'=>'id', 'nr_cep'))
            ->joinLeft(['sc' => TB_CIDADE], 'e.cd_cidade = sc.id', ['cdIbge' => 'sc.cd_ibge', 'ufIbge' => 'sc.uf']);

        if ($idRespJuridico) {
            $contratosPF->joinLeft(array('ec' => TB_EMPREENDIMENTO_CONTABEIS), 'ec.id_empreendimento = e.id', array('vl_porc_declarada'))
                ->where('id_pessoa_juridica = '.$idRespJuridico);
        }

        $contratosPF = $contratosPF->where('cp.id_contrato IN ("' . $contratos.'")')
            ->group(array('cp.id_pessoa', 'cp.id_contrato'))->query()->fetchAll();
       
        $contratosPJ = Zend_Db_Table::getDefaultAdapter()->select()
            ->from(array('l' => TB_LOTES), array('lote', 'quadra'))
            ->join(array('c' => TB_CONTRATO), 'l.id = c.id_lote', array('nr_parcela', 'vl_parcela', 'nr_parcela_sinal', 'vl_sinal', 'nr_intercalada', 
                'vl_intercalada', 'nr_parcela_entrega', 'vl_parcela_entrega', 'dt_contrato', 'inclui_sinal_contrato', 'fl_distrato', 'fl_itens_contrato'))
            ->join(['cp' => TB_CONTRATO_PESSOA], 'c.id = cp.id_contrato', ['id_contrato', 'porcentagem_participacao_contrato'])
            ->join(array('pe' => TB_PESSOA), 'pe.id = cp.id_pessoa', array('nm_pessoa', 'idPessoa' => 'id'))
            ->join(array('pj' => TB_PESSOA_JURIDICA), 'pe.id = pj.id_pessoa', array('cpf_cnpj' => 'nr_cnpj'))
            ->join(array('e' => TB_EMPREENDIMENTO), 'e.id = l.id_empreendimento', array('nm_empreendimento','idEmpreendimento'=>'id', 'nr_cep'))
            ->joinLeft(['sc' => TB_CIDADE], 'e.cd_cidade = sc.id', ['cdIbge' => 'sc.cd_ibge', 'ufIbge' => 'sc.uf']);

        if ($idRespJuridico) {
            $contratosPJ->joinLeft(array('ec' => TB_EMPREENDIMENTO_CONTABEIS), 'ec.id_empreendimento = e.id', array('vl_porc_declarada'))
                ->where('id_pessoa_juridica = '.$idRespJuridico);
        }

        $contratosPJ = $contratosPJ->where('cp.id_contrato IN ("' . $contratos.'")')
            ->group('cp.id_pessoa')->query()->fetchAll();
            
        return array_merge($contratosPF, $contratosPJ);
    }

    public function buscaOutrosCompradoresSped($contratos, $idRespJuridico)
    {
        $contratos = implode('","', $contratos);

        $contratosPF = Zend_Db_Table::getDefaultAdapter()->select()
            ->from(array('l' => TB_LOTES), array('lote', 'quadra'))
            ->join(array('c' => TB_CONTRATO), 'l.id = c.id_lote', array('nr_parcela',
                'vl_parcela', 'nr_parcela_sinal', 'vl_sinal', 'nr_intercalada', 'vl_intercalada',
                'nr_parcela_entrega', 'vl_parcela_entrega', 'dt_contrato', 'inclui_sinal_contrato', 'fl_distrato', 'fl_itens_contrato'))
            ->join(['cp' => TB_CONTRATO_PESSOA], 'c.id = cp.id_contrato', ['id_contrato', 'porcentagem_participacao_contrato'])
            ->join(array('pe' => TB_PESSOA), 'pe.id = cp.id_pessoa', array('nm_pessoa', 'idPessoa' => 'id'))
            ->join(array('pf' => TB_PESSOA_FISICA), 'pe.id = pf.id_pessoa', array('cpf_cnpj' => 'nr_cpf'))
            ->join(array('e' => TB_EMPREENDIMENTO), 'e.id = l.id_empreendimento', array('nm_empreendimento','idEmpreendimento'=>'id'));

        if ($idRespJuridico) {
            $contratosPF->joinLeft(array('ec' => TB_EMPREENDIMENTO_CONTABEIS), 'ec.id_empreendimento = e.id', array('vl_porc_declarada'))
                ->where('id_pessoa_juridica = '.$idRespJuridico);
        }

        $contratosPF = $contratosPF->where('cp.id_contrato IN ("' . $contratos.'")')
            ->group(array('cp.id_pessoa', 'cp.id_contrato'))->query()->fetchAll();
       
        $contratosPJ = Zend_Db_Table::getDefaultAdapter()->select()
            ->from(array('l' => TB_LOTES), array('lote', 'quadra'))
            ->join(array('c' => TB_CONTRATO), 'l.id = c.id_lote', array('nr_parcela', 'vl_parcela', 'nr_parcela_sinal', 'vl_sinal', 'nr_intercalada', 
                'vl_intercalada', 'nr_parcela_entrega', 'vl_parcela_entrega', 'dt_contrato', 'inclui_sinal_contrato', 'fl_distrato', 'fl_itens_contrato'))
            ->join(['cp' => TB_CONTRATO_PESSOA], 'c.id = cp.id_contrato', ['id_contrato', 'porcentagem_participacao_contrato'])
            ->join(array('pe' => TB_PESSOA), 'pe.id = cp.id_pessoa', array('nm_pessoa', 'idPessoa' => 'id'))
            ->join(array('pj' => TB_PESSOA_JURIDICA), 'pe.id = pj.id_pessoa', array('cpf_cnpj' => 'nr_cnpj'))
            ->join(array('e' => TB_EMPREENDIMENTO), 'e.id = l.id_empreendimento', array('nm_empreendimento','idEmpreendimento'=>'id'));

        if ($idRespJuridico) {
            $contratosPJ->joinLeft(array('ec' => TB_EMPREENDIMENTO_CONTABEIS), 'ec.id_empreendimento = e.id', array('vl_porc_declarada'))
                ->where('id_pessoa_juridica = '.$idRespJuridico);
        }

        $contratosPJ = $contratosPJ->where('cp.id_contrato IN ("' . $contratos.'")')
            ->group('cp.id_pessoa')->query()->fetchAll();
            
        return array_merge($contratosPF, $contratosPJ);
    }

    public function getContratoById($id)
    {
        return $this->fetchRow('id = ' . $id);
    }

    public function getTabelaPrice($idContrato){
        //busca o banco atravz do id do contrato.
        $contrato = new Contrato();
        $contrato = $contrato->getContrato($idContrato);

        //Checa se o contrato possui enquadramento na tabela Price
        $checkPrice = ($contrato['vl_divida'] && $contrato['nr_meses_financiamento'] && $contrato['tx_juros'])?true:false;

        if($checkPrice){
            return $contrato['tx_juros'];
        }

        return null;
    }
}